import logging
import os
import re
from collections import namedtuple

from sandbox import sdk2

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.market.wms.helpers import build_utils as bu
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.projects.common.gnupg import GpgKey2
from sandbox.projects.common.debpkg import DebRelease
from sandbox.sdk2.vcs.git import Git
from sandbox.projects.resource_types import OTHER_RESOURCE as OtherResource
from sandbox.projects import resource_types as rt


from sandbox.projects.market.wms import modules

logger = logging.getLogger(__name__)

ROBOT_USER = 'robot-market-packer'
DUPLOAD_CONF = {'market-common': {}}

DebFile = namedtuple('DebFile', ['name', 'version', 'path'])


class WmsBuildDeb(sdk2.Task):
    work_dir_name = 'wms'

    class Requirements(sdk2.Task.Requirements):
        environments = (
            SandboxJavaJdkEnvironment('11.0.2'),
        )
        client_tags = ctc.Tag.Group.LINUX
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Git Settings') as git_block:
            repository = sdk2.parameters.String("Ssh git repository", required=True)
            branch = sdk2.parameters.String("Release branch", required=True, default="master")
            commit = sdk2.parameters.String("Release commit")
            stable = sdk2.parameters.String("Last stable commit")
            git_ssh = sdk2.parameters.YavSecret("Ssh key", default="sec-01dgd1q7tdynkch7sdcf1taxyt", required=True)

        with sdk2.parameters.Group('Release Settings') as release_block:
            version = sdk2.parameters.String("Fixed version", default='0.0.0-SNAPSHOT')

        with sdk2.parameters.Group('Build Settings') as build_block:
            excluded_tasks = sdk2.parameters.List("Excluded Gradle tasks", required=False)

    class Context(sdk2.Task.Context):
        execution_number = 0

    def on_execute(self):
        self.Context.execution_number += 1
        logger.info("Clone git repository {repository}".format(
            repository=self.Parameters.repository
        ))
        ssh_key = self.Parameters.git_ssh.data()['ssh-private']
        with sdk2.ssh.Key(self, private_part=ssh_key):
            git = Git(self.Parameters.repository)
            git.clone(target_dir=self.work_dir_name, branch=self.Parameters.branch,
                      commit=self.Parameters.commit)
        try:
            self.do_build()
            self.do_publish()
            self.create_resources()
        finally:
            self._collect_and_publish_reports()

    def do_build(self):
        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            cmd = ['./gradlew',
                    '-Dorg.gradle.project.fixedVersion=true', '-Dorg.gradle.project.version={}'.format(self.Parameters.version),
                    '-Pinfor.release.type=DEB',
                    '-Previsions.stable={}'.format(self.Parameters.stable),
                    '--parallel',
                    '--stacktrace',
                    'clean', 'buildDeb',
                    '-x', 'test']

            for excluded_task in self.Parameters.excluded_tasks:
                cmd += ['-x', excluded_task]

            sp.check_call(cmd, cwd=self.work_dir_name, stdout=pl.stdout, stderr=pl.stderr)
            self.set_info('Modules have been built')

    def create_resources(self):
        for deb in self._find_all_debs():
            logger.info('creating resource with name={} and version={}'.format(deb.name, deb.version))
            self._create_resource(deb.name, deb.version, deb.path)

    def _create_resource(self, name, version, file_path):
        resource = OtherResource(self, '', file_path, resource_name=name, resource_version=version)
        data = sdk2.ResourceData(resource)
        data.ready()
        logger.info('resource {} created'.format(resource.id))

    def do_publish(self):
        for deb in self._find_all_debs():
            changes = deb.path.replace('.deb', '.changes')
            with GpgKey2('MARKETSRE', '{}-gpg-private'.format(ROBOT_USER), '{}-gpg-public'.format(ROBOT_USER)):
                with sdk2.helpers.ProcessLog(self, logger='publish') as pl:
                    sp.check_call(['debsign', '-k{}@yandex-team.ru'.format(ROBOT_USER),
                                   changes], stdout=pl.stdout, stderr=pl.stderr)
                    with sdk2.ssh.Key('MARKETSRE', '{}-ssh'.format(ROBOT_USER)):
                        with DebRelease(DUPLOAD_CONF, ROBOT_USER):
                            sp.check_call(['dupload', '--to', 'market-common', changes], stdout=pl.stdout,
                                          stderr=pl.stderr)

    def _find_all_debs(self):
        package_pattern = re.compile(r'(.*)_(.*)_.*\.deb')
        for module in [module for module in modules.all_modules if module.nebula]:
            dist_dir = os.path.join(self.work_dir_name, module.work_dir, 'build/distributions')
            logger.info('finding deb files in dist dir {}'.format(dist_dir))
            for root, _, files in os.walk(dist_dir):
                for f_file in files:
                    match = package_pattern.match(f_file)
                    if match and len(match.groups()) == 2:
                        logger.info('founded deb file: {}'.format(files))
                        yield DebFile(match.group(1), match.group(2), os.path.join(root, f_file))

    def _collect_and_publish_reports(self):
        reports_path = "build_reports{}".format(self.Context.execution_number)
        bu.copy_reports(self.work_dir_name, reports_path)

        if (os.path.exists(reports_path)):
            resource = rt.BUILD_OUTPUT(self, 'Build reports', reports_path, resource_name='build-reports')
            data = sdk2.ResourceData(resource)
            data.ready()
            logger.info('Resource {} created'.format(resource.id))
