# coding=utf-8

import logging
import os.path
import json

import sandbox.common.types.misc as ctm

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.common.types import resource as ctr
from sandbox.projects.common.arcadia import sdk as arc
from sandbox.projects.common.build.parameters import ArcadiaUrl


class MARKET_WMS_MT_BALANCERS_JSON(sdk2.Resource):
    releasable = False
    executable = False
    releasers = ["MARKET"]


class WmsGenerateBalancerJsonMT(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        cores = 1
        ram = 512

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        checkout_arcadia_from_url = ArcadiaUrl()
        template_parameters = sdk2.parameters.String("Параметры шаблона (json)", required=True)
        template_path = sdk2.parameters.String("Путь до шаблона", required=True)
        template_name = sdk2.parameters.String("Название шаблона", required=True)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {
                "task_type": "WMS_GENERATE_BALANCER_JSON_MT",
                "released": self.Parameters.binary_executor_release_type
            },
            "state": [ctr.State.READY]
        }

    def _get_arcadia(self):
        return arc.mount_arc_path(self.Parameters.checkout_arcadia_from_url)

    def on_execute(self):
        super(WmsGenerateBalancerJsonMT, self).on_execute()

        params = json.loads(self.Parameters.template_parameters)
        result = self.render_template(params)
        result_path = self.save_to_file(result)

        sdk2.ResourceData(MARKET_WMS_MT_BALANCERS_JSON(
            self, "Balancer json for wms multitestings", result_path
        )).ready()

    def render_template(self, params):
        import jinja2

        with self._get_arcadia() as arcadia_src_dir:
            template_path = os.path.join(arcadia_src_dir, self.Parameters.template_path, self.Parameters.template_name)
            logging.info("Template path: " + template_path)

            with open(template_path, 'r') as template:
                tmp = template.read()
                logging.info(tmp)

                t = jinja2.Template(tmp)
                result = t.render(**params)
                logging.info("Rendered template:\n" + result)
                template.close()
        return result

    def save_to_file(self, content):
        result_file = os.path.join(self.Parameters.template_name)

        with open(result_file, 'w+') as target:
            target.write(content)
            target.close()

        return result_file
