# coding: utf-8
import logging
import os

from sandbox import sdk2

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.arcadia import sdk as arcadiasdk

WMS_LOAD_FOLDER = "market/logistics/wms-load"

logger = logging.getLogger(__name__)


class WmsLoadTesting(sdk2.Task):
    """
    WMS load testing
    """
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024
        disk_space = 1024
        client_tags = ctc.Tag.Group.LINUX
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        enable_yav = True

        with sdk2.parameters.Group("Arc Settings") as arc_block:
            revision = sdk2.parameters.String("Arc revision", required=False, default=None)

        with sdk2.parameters.Group("Env Settings") as env_block:
            db_secret = sdk2.parameters.YavSecret("DB credentials secret", required=True,
                                                  default="sec-01eaa02anhkadt5n8h8p75p1ce")

        with sdk2.parameters.Group('Load Testing Settings') as load_testing_block:
            suspend_after_tests = sdk2.parameters.Bool("Suspend after tests", default=False)

            config = sdk2.parameters.String("Config", required=True, default=None)

            with sdk2.parameters.RadioGroup("Run test in:") as mode:
                mode.values["local"] = mode.Value("Sandbox")
                mode.values["remote"] = mode.Value("Tank", default=True)

            with mode.value["remote"]:
                ticket = sdk2.parameters.String("Ticket", required=True, default=None)
                ya_token = sdk2.parameters.YavSecretWithKey("YA token", required=False)
                ya_token_owner_group = sdk2.parameters.String("User (or token owner) Sandbox group", required=False)

    def _format_command(self):
        ticket = self.Parameters.ticket
        mode = self.Parameters.mode
        config = self.Parameters.config

        if mode == 'local':
            return './run_load_test.sh -l %s' % (config)
        if mode == 'remote':
            return './run_load_test.sh -r %s %s' % (config, ticket)

        raise Exception('Unexpected mode value %s' % (mode))

    def _make_environ(self):
        current_envs = dict(os.environ)
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_WMS_USER', 'WmsUser')
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_WMS_PASS', 'WmsPassword')
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_WMS_WRAP_INFOR_TOKEN', 'wrap-infor-token')
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_WMS_DB_HOST', 'DbHost')
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_WMS_DB_PORT', 'DbPort')
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_WMS_DB_USER', 'DbUser')
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_WMS_DB_PASS', 'DbPassword')
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_SOLOMON_TOKEN', 'SolomonToken')
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_STATS_DB_ADDR', 'StatsDbAddr')
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_STATS_DB_USER', 'StatsDbUser')
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_STATS_DB_PASS', 'StatsDbPassword')
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_SHIPPINGSORTER_DB_PASS', 'PostgresDbShippingsorterPass')
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_SHIPPINGSORTER_DB_USER', 'PostgresDbShippingsorterUser')
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_SHIPPINGSORTER_DB_HOST', 'PostgresDbHost')
        self._add_secret_entry(current_envs,
                               'LOAD_TESTING_SHIPPINSORTER_DB_PORT', 'PostgresDbPort')

        load_tmp_path = os.path.join(os.getcwd(), "tmp", "wms_load", "conf")
        current_envs["WMS_LOAD_TMP_DIR"] = load_tmp_path

        load_logs_path = os.path.join(self.agentr.logdir, "fire.log")
        current_envs["WMS_LOAD_TESTING_LOG"] = load_logs_path

        gen_logs_path = os.path.join(self.agentr.logdir, "gen.log")
        current_envs["WMS_LOAD_GEN_LOG"] = gen_logs_path

        current_envs["WMS_LOAD_USER"] = self.author

        ya_token_value = None
        if self.Parameters.ya_token:
            ya_token_value = self.Parameters.ya_token.data()[self.Parameters.ya_token.default_key]
        current_envs["WMS_LOAD_SB_TOKEN"] = ya_token_value or self.server.task_token

        current_envs["WMS_LOAD_SB_OWNER_GROUP"] = self.Parameters.ya_token_owner_group or self.owner

        return current_envs

    def _add_secret_entry(self, env, env_entry_name, secret_entry_name):
        env[env_entry_name] = self.Parameters.db_secret.data()[secret_entry_name]

    def _run_test(self, repo_root):
        with sdk2.helpers.ProcessLog(self, logger='load-test') as pl:
            sp.check_call([self._format_command()],
                          shell=True,
                          stdout=pl.stdout,
                          stderr=pl.stderr,
                          env=self._make_environ(),
                          cwd=os.path.join(repo_root, WMS_LOAD_FOLDER))

    def on_execute(self):
        arcadia_url = sdk2.svn.Arcadia.ARCADIA_TRUNK_URL
        revision = self.Parameters.revision
        if revision:
            arcadia_url += "@" + revision
        with arcadiasdk.mount_arc_path(arcadia_url) as repo_root:
            try:
                self._run_test(repo_root)
            finally:
                if self.Parameters.suspend_after_tests:
                    self.suspend()
