# coding: utf-8
import logging
import os
import shutil

from sandbox import sdk2

import sandbox.common.types.client as ctc
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.sdk2.vcs.git import Git
from sandbox.projects.market.wms.resources import resources as wms
from sandbox.sdk2.helpers import subprocess as sp

MIGRATION_TARGET_PATH = "./infor/db-migrations/db"
GENERAL_USER_MIGRATION_TARGET_PATH = "./infor/db-migrations/users"
USER_MIGRATION_TARGET_PATH = "./infor/db-migrations/users/db"
MIGRATION_SOURCE_PATH = "./infor/db-migrations"
MASTER_FILE = "changelog-master.xml"
CHECKOUT_FOLDER = "infor"
MS_SQL_JDBC_URL = "jdbc:jtds:sqlserver://{db_url}:{db_port}/{scheme}"

logger = logging.getLogger(__name__)


class WmsMigrations(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            SandboxJavaJdkEnvironment('17'),
        )
        client_tags = ctc.Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):

        dry_run = sdk2.parameters.Bool("Dry run, don't start migration", default=False)
        debug_mode = sdk2.parameters.Bool("Debug mode", default=False)
        release_version = sdk2.parameters.String("Release version", default="1.0")

        with sdk2.parameters.Group('Git Settings') as git_block:
            repository = sdk2.parameters.String("Ssh git repository", required=True)
            branch = sdk2.parameters.String("Release branch")
            commit = sdk2.parameters.String("Release commit")
            git_ssh = sdk2.parameters.YavSecret("Ssh key", default="sec-01dgd1q7tdynkch7sdcf1taxyt",
                                                required=True)

        with sdk2.parameters.Group('Database Settings') as settings_block:
            db_url = sdk2.parameters.String("Database url", required=True)
            db_port = sdk2.parameters.String("Database port", required=True)
            db_secret = sdk2.parameters.YavSecret("Database secret", required=True)

        with sdk2.parameters.Group('Liquibase Settings') as liquibase_block:
            liquibase_resource = sdk2.parameters.Resource("Liquibase resource",
                                                          resource_type=wms.WMS_LIQUIBASE,
                                                          required=True,
                                                          default_value=2106885217)
            with sdk2.parameters.String("Liquibase command", multiline=True) as command:
                command.values.update = command.Value(default=True)
                command.values.rollback = None
            tag = sdk2.parameters.String("Rollback tag")
            options = sdk2.parameters.List("Liquibase options",
                                           required=True,
                                           default=["--logLevel=debug"])

    class MigrationData:

        def __init__(self, liquibase_path, db_url, db_port, username, password, command, options, dry_run, tag):
            self.liquibase = "{path}/{file}".format(path=liquibase_path, file="liquibase")
            self.db_url = db_url
            self.db_port = db_port
            self.username = username
            self.password = password
            self.command = command
            self.options = options
            self.dry_run = dry_run
            self.tag = tag

    def prepare_folders(self):
        logger.info("Prepare folders")
        for scheme_name in ["SCPRD", "SCPRDMST"]:
            self._copy_migrations(scheme_name)

        logger.info("Folders layout in {path}:".format(
            path=MIGRATION_TARGET_PATH
        ))

        folders = os.listdir(MIGRATION_TARGET_PATH)
        logger.info("\n".join(folders))

    def _make_dir(self, path):
        self._remove_dir(path)
        os.mkdir(path)

    def _copy_migrations(self, scheme):
        target_scheme_path = '{path}/{scheme}'.format(
            path=MIGRATION_TARGET_PATH,
            scheme=scheme
        )

        source_migrations_path = '{path}/migrations'.format(
            path=MIGRATION_SOURCE_PATH
        )

        target_migrations_path = '{path}/{db_scheme}/migrations'.format(
            path=MIGRATION_TARGET_PATH,
            db_scheme=scheme
        )

        source_master_file_path = "{path}/{file}".format(
            path=MIGRATION_SOURCE_PATH,
            file=MASTER_FILE
        )

        target_master_file_path = "{path}/{scheme}/{file}".format(
            path=MIGRATION_TARGET_PATH,
            scheme=scheme,
            file=MASTER_FILE
        )

        self._make_dir(target_scheme_path)
        shutil.copytree(source_migrations_path, target_migrations_path)
        shutil.copyfile(source_master_file_path, target_master_file_path)

    @staticmethod
    def _remove_dir(path):
        if os.path.exists(path):
            shutil.rmtree(path)

    def do_migrations(self):
        logger.info("Start migration process")

        migration_data = self.MigrationData(str(sdk2.ResourceData(self.Parameters.liquibase_resource).path),
                                            self.Parameters.db_url,
                                            self.Parameters.db_port,
                                            self.Parameters.db_secret.data()['DbWmsAdminUser'],
                                            self.Parameters.db_secret.data()['DbWmsAdminPass'],
                                            self.Parameters.command,
                                            self.Parameters.options,
                                            self.Parameters.dry_run,
                                            self.Parameters.tag)

        for migration_dir in os.listdir(MIGRATION_TARGET_PATH):
            self._do_migration(migration_dir, migration_data, MIGRATION_TARGET_PATH)

        self.suspend_if_needed()
        if os.path.isdir(USER_MIGRATION_TARGET_PATH):
            for migration_dir in os.listdir(USER_MIGRATION_TARGET_PATH):
                self._do_migration(migration_dir, migration_data, USER_MIGRATION_TARGET_PATH)

    def _do_migration(self, migration_dir, migration_data, migration_path):
        liquibase = migration_data.liquibase
        db_url = migration_data.db_url
        db_port = migration_data.db_port
        username = migration_data.username
        password = migration_data.password
        command = migration_data.command
        options = migration_data.options
        dry_run = migration_data.dry_run
        tag = migration_data.tag

        jdbc_url = self._format_jdbc_url(db_url, db_port, migration_dir)
        liquibase_command = self._format_liquibase_command(liquibase, migration_path, migration_dir,
                                                           jdbc_url, username, password, command, options, tag)
        if dry_run:
            logger.info("Executing liquibase command:\n")
            logger.info(liquibase_command)
            logger.info("Database {migration_dir} updated".format(
                migration_dir=migration_dir
            ))
            return

        with sdk2.helpers.ProcessLog(self, logger='liquibase') as pl:
            sp.check_call([liquibase_command], shell=True, stdout=pl.stdout, stderr=pl.stderr,
                          env=self._make_environ())
        logger.info("Database {migration_dir} updated".format(
            migration_dir=migration_dir
        ))

    def _make_environ(self):
        current_envs = dict(os.environ)
        current_envs.update(self.Parameters.db_secret.data())
        return current_envs

    @staticmethod
    def _format_liquibase_command(liquibase_path, classpath, migration_dir, jdbc_url, username,
                                  password, command, options, arguments):

        if migration_dir in ["SCPRD", "SCPRDMST"]:
            master_file = MASTER_FILE
            classpath = "{classpath}/{folder}".format(classpath=classpath, folder=migration_dir)
        else:
            master_file = "{folder}/{master_file}".format(folder=migration_dir, master_file=MASTER_FILE)
        return '{liquibase_path} --driver=net.sourceforge.jtds.jdbc.Driver ' \
               '--classpath={classpath} --changeLogFile={file} ' \
               '--url="{jdbc_url}" --username={username} --password={password} {options} {cmd} {arguments}' \
            .format(liquibase_path=liquibase_path,
                    classpath=classpath,
                    file=master_file,
                    jdbc_url=jdbc_url,
                    username=username,
                    password=password,
                    cmd=command,
                    options=" ".join(options),
                    arguments=arguments or ''
                    )

    @staticmethod
    def _format_jdbc_url(url, port, scheme):
        return MS_SQL_JDBC_URL.format(db_url=url, db_port=port, scheme=scheme)

    @staticmethod
    def _get_resource_path(resource_id):
        return str(sdk2.ResourceData(sdk2.Resource.find(id=resource_id).first()).path)

    def suspend_if_needed(self):
        if self.Parameters.debug_mode:
            self.suspend()

    def on_execute(self):
        self.suspend_if_needed()

        if self.Parameters.dry_run:
            logger.info("WARNING! Script started with dry run mode, migrations won't be executed!")

        logger.info("Clone git repository {repository}".format(
            repository=self.Parameters.repository
        ))
        ssh_key = self.Parameters.git_ssh.data()['ssh-private']
        with sdk2.ssh.Key(self, private_part=ssh_key):
            git = Git(self.Parameters.repository)
            git.clone(target_dir=CHECKOUT_FOLDER, branch=self.Parameters.branch, commit=self.Parameters.commit)
            self.suspend_if_needed()

            self.prepare_folders()
            self.suspend_if_needed()
            self.do_migrations()
            self.suspend_if_needed()
