import logging
import os

from sandbox import sdk2

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.projects.market.wms.helpers import build_utils as bu
from sandbox.sdk2.vcs.git import Git

from sandbox.projects.market.wms import modules
from sandbox.projects import resource_types as rt

logger = logging.getLogger(__name__)


class WmsRunUnitTests(sdk2.Task):
    work_dir_name = 'wms'

    class Requirements(sdk2.Task.Requirements):
        environments = (
            SandboxJavaJdkEnvironment('11.0.2'),
        )
        client_tags = ctc.Tag.Group.LINUX
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Git Settings') as git_block:
            repository = sdk2.parameters.String("Ssh git repository", required=True)
            branch = sdk2.parameters.String("Release branch", required=True, default="master")
            commit = sdk2.parameters.String("Release commit")
            git_ssh = sdk2.parameters.YavSecret("Ssh key", default="sec-01dgd1q7tdynkch7sdcf1taxyt", required=True)

    class Context(sdk2.Task.Context):
        execution_number = 0

    def on_execute(self):
        self.Context.execution_number += 1
        logger.info("Clone git repository {repository}".format(
            repository=self.Parameters.repository
        ))
        ssh_key = self.Parameters.git_ssh.data()['ssh-private']
        with sdk2.ssh.Key(self, private_part=ssh_key):
            git = Git(self.Parameters.repository)
            git.clone(target_dir=self.work_dir_name, branch=self.Parameters.branch,
                      commit=self.Parameters.commit)
        try:
            self.do_tests()
        finally:
            self._collect_and_publish_reports()

    def do_tests(self):
        for module in modules.all_modules:
            with sdk2.helpers.ProcessLog(self, logger=module.name) as pl:
                sp.check_call(['./gradlew', '{}:test'.format(module.gradle)],
                              cwd=self.work_dir_name, stdout=pl.stdout, stderr=pl.stderr)
                self.set_info('{} module has been tested.'.format(module.name))

    def _collect_and_publish_reports(self):
        reports_path = "build_reports{}".format(self.Context.execution_number)
        bu.copy_reports(self.work_dir_name, reports_path)

        if (os.path.exists(reports_path)):
            resource = rt.BUILD_OUTPUT(self, 'Build reports', reports_path, resource_name='build-reports')
            data = sdk2.ResourceData(resource)
            data.ready()
            logger.info('Resource {} created'.format(resource.id))
