# coding=utf-8

import json
import logging

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.projects.sandbox_ci.utils.request import send_request
from sandbox.sdk2 import Task
from sandbox.projects.common import binary_task
from sandbox.common.types import resource as ctr
from sandbox.projects.market.wms.utils_py3 import mt_utils

NOTIFICATION = """**Запустился пайплайн по созданию мультитестинговой среды**
        Ссылка на среду: {}
        Ссылка на ui: {}
        Ссылка на страницу статусов: {}
        Инструкция по мультитестингу: https://wiki.yandex-team.ru/delivery/fulfilment/wms/infra/wms-multitestingi/
        """


class WmsStartMT(binary_task.LastBinaryTaskRelease, Task):
    """
    Таска для запуска и перезапуска мультитестинга wms.
    """

    class Requirements(Task.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.MULTISLOT

    class Parameters(Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Output():
            mt_url = sdk2.parameters.String("Multitesting url")

        ticket_id = sdk2.parameters.String("Тикет", required=True)
        branch = sdk2.parameters.String("Ветка")
        arcadia_url = sdk2.parameters.ArcadiaUrl("Arcadia repository url", default="arcadia-arc:/#trunk", required=True)
        tsum_project = sdk2.parameters.String("Название проекта", default="wms", required=True)
        tsum_pipeline_type = sdk2.parameters.String("Тип пайплайна", default="wms-autotests-mt", required=True)
        robot_secret = sdk2.parameters.YavSecret("Robot secret", default="sec-01dgd1q7tdynkch7sdcf1taxyt",
                                                 required=True)
        arcanum_review_id = sdk2.parameters.Integer("Arcanum PR id", default=None)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {
                "task_type": "WMS_START_MT",
                "released": self.Parameters.binary_executor_release_type
            },
            "state": [ctr.State.READY]
        }

    @property
    def _tsum_mt_name(self):
        return mt_utils.tsum_mt_name(self.Parameters.ticket_id, self.Parameters.branch)

    @property
    def _issue_name(self):
        return self.Parameters.ticket_id

    @property
    def _tsum_api_url(self):
        return mt_utils.tsum_api_url(self.Parameters.tsum_project, self._tsum_mt_name)

    @property
    def _tsum_page_url(self):
        return mt_utils.tsum_page_url(self.Parameters.tsum_project, self._tsum_mt_name)

    @property
    def _tsum_exists_mt_page_url(self):
        return mt_utils.tsum_exists_mt_page_url(self.Parameters.tsum_project, self._tsum_mt_name)

    @property
    def _robot_secret(self):
        return self.Parameters.robot_secret

    @property
    def _tsum_headers(self):
        return {
            "Authorization": self._robot_secret.data()["oauth-token-tsum"],
            "Content-Type": "application/json"
        }

    @property
    def _balancer_url(self):
        return mt_utils.balancer_ui_url(self.Parameters.tsum_project, self._tsum_mt_name)

    @property
    def _status_url(self):
        return "{}/status".format(mt_utils.balancer_url(self.Parameters.tsum_project, self._tsum_mt_name))

    def _notify_pr(self, message):
        if not self.Parameters.arcanum_review_id:
            return

        import vcs.arcanum as arcanum
        import vcs.arcanum.comments

        arcanum_client = arcanum.Arcanum(auth=self._robot_secret.data()["oauth-token-arcanum"])

        comments_api = arcanum.comments.CommentsApi(arcanum_client)
        if self.Parameters.arcanum_review_id:
            comments_api.post_review_requests_comments(
                review_request_id=self.Parameters.arcanum_review_id,
                content=message,
                draft=False,
            )

    def _send_startrek_report(self, message):
        try:
            from startrek_client import Startrek
            st = Startrek(useragent="robot-wms-robokotov", token=self._robot_secret.data()["star-trek-token"])
            issue = st.issues[self._issue_name]
            issue.comments.create(text=message)
        except Exception as e:
            logging.error(e.message)

        return

    def _start_mt(self):
        data = {
            "name": self._tsum_mt_name,
            "type": "USE_EXISTING_PIPELINE",
            "pipelineId": self.Parameters.tsum_pipeline_type,
            "resources": {
                "fd4f152f-9f11-4b91-8a35-e58503b6bdf6": {
                    "tickets": [self.Parameters.ticket_id]
                },
                "c549477a-867a-483f-947b-5fd64feef256": {
                    "ref": self.Parameters.arcadia_url
                },
                "7742cc9f-6941-446f-8900-3f9c039917dd": {
                    "name": "master"
                }
            }
        }

        res = send_request("post", self._tsum_api_url, headers=self._tsum_headers, data=json.dumps(data))
        res.raise_for_status()
        return res

    def _has_mt(self):
        res = send_request("get", self._tsum_exists_mt_page_url, headers=self._tsum_headers)
        res.raise_for_status()

        return res.text == "true"

    def create_or_restart_mt(self):
        self.Parameters.mt_url = mt_utils.balancer_url(self.Parameters.tsum_project, self._tsum_mt_name)

        message = NOTIFICATION.format(self._tsum_page_url, self._balancer_url, self._status_url)

        has_mt = self._has_mt()
        res = self._start_mt()
        try:
            assert res.json()["name"] == self._tsum_mt_name
            if not has_mt:
                self._notify_pr(message)
                self._send_startrek_report(message)
        except Exception as e:
            logging.error(e.message)
            logging.error(u"Bad TSUM response: {}".format(res.text))
            self._send_startrek_report("Ошибка при запуске среды: {}".format(res.text))

            raise Exception("Something is wrong, see logs")

    def on_execute(self):
        self.create_or_restart_mt()
