# coding=utf-8

import logging
import json

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.sdk2 import Task
from sandbox.projects.common import binary_task
from sandbox.projects.sandbox_ci.utils.request import send_request
from sandbox.common.types import resource as ctr
from sandbox.projects.market.wms.utils_py3 import mt_utils

STARTREK_ISSUE = "https://st-api.yandex-team.ru/v2/issues/{issue}"


class WmsStopMT(binary_task.LastBinaryTaskRelease, Task):
    class Requirements(Task.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.MULTISLOT

    class Parameters(Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        ticket_id = sdk2.parameters.String("Тикет", required=True)
        branch = sdk2.parameters.String("Ветка")

        tsum_project = sdk2.parameters.String("Название проекта", default_value="wms", required=True)
        startrek_token = sdk2.parameters.YavSecret("Robot startrek token", required=True,
                                                   default="sec-01dgd1q7tdynkch7sdcf1taxyt#tsum.startrek.auth-token")
        tsum_token = sdk2.parameters.YavSecret("Robot tsum token", required=True,
                                               default="sec-01dgd1q7tdynkch7sdcf1taxyt#oauth-token-tsum")

    @property
    def binary_executor_query(self):
        return {
            "attrs": {
                "task_type": "WMS_STOP_MT",
                "released": self.Parameters.binary_executor_release_type
            },
            "state": [ctr.State.READY]
        }

    @property
    def startrek_token(self):
        return self.Parameters.startrek_token.data()[self.Parameters.startrek_token.default_key]

    @property
    def tsum_token(self):
        return self.Parameters.tsum_token.data()[self.Parameters.tsum_token.default_key]

    @property
    def _issue_url(self):
        return STARTREK_ISSUE.format(issue=self.Parameters.ticket_id)

    @property
    def _tsum_mt_name(self):
        return mt_utils.tsum_mt_name(self.Parameters.ticket_id, self.Parameters.branch)

    @property
    def _tsum_archive_url(self):
        return mt_utils.tsum_archive_url(self.Parameters.tsum_project, self._tsum_mt_name)

    def has_manual_testing(self):
        headers = {
            "Authorization": "OAuth {token}".format(token=self.startrek_token)
        }
        res = send_request("get", self._issue_url, headers=headers)
        res.raise_for_status()
        logging.info(res)
        content = json.loads(res.text)
        if "manualTesting" not in content:
            return False

        return content["manualTesting"] == 'Требуется проверка'

    def stop_mt(self):
        headers = {
            "Authorization": self.tsum_token,
            "Content-Type": "application/json"
        }
        send_request('post', self._tsum_archive_url, headers=headers)

    def on_execute(self):
        super(WmsStopMT, self).on_execute()
        if not self.has_manual_testing():
            self.stop_mt()
