# coding=utf-8

import logging
import time

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.sdk2 import Task
from sandbox.projects.sandbox_ci.utils.request import send_request
from sandbox.projects.common import binary_task
from sandbox.common.types import resource as ctr
from sandbox.projects.market.wms.utils_py3 import mt_utils
from sandbox.projects.market.wms.utils_py3.mt_utils import DeployStatus


class WmsWaitMTAutotests(binary_task.LastBinaryTaskRelease, Task):
    """
    Таска для запуска и перезапуска мультитестинга wms.
    """

    class Requirements(Task.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.MULTISLOT

    class Parameters(Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Output():
            mt_url = sdk2.parameters.String("Multitesting url")

        ticket_id = sdk2.parameters.String("Тикет", required=True)
        branch = sdk2.parameters.String("Ветка")
        tsum_project = sdk2.parameters.String("Название проекта", default_value="wms", required=True)
        robot_secret = sdk2.parameters.YavSecret("Robot secret", required=True,
                                                 default="sec-01dgd1q7tdynkch7sdcf1taxyt")

    @property
    def binary_executor_query(self):
        return {
            "attrs": {
                "task_type": "WMS_WAIT_MTAUTOTESTS",
                "released": self.Parameters.binary_executor_release_type
            },
            "state": [ctr.State.READY]
        }

    @property
    def _tsum_mt_name(self):
        return mt_utils.tsum_mt_name(self.Parameters.ticket_id, self.Parameters.branch)

    @property
    def _tsum_mt_status(self):
        return mt_utils.tsum_mt_status(self.Parameters.tsum_project, self._tsum_mt_name)

    @property
    def _robot_secret(self):
        return self.Parameters.robot_secret

    def wait_for_deploy(self):

        # Wait for changing deploy status
        time.sleep(60)

        tsum_pipeline_retry = 5
        deploy_status = DeployStatus.DEPLOYING.value

        # Give a chance for tsum jobs for retry
        while tsum_pipeline_retry > 0:

            deploy_status = self._get_mt_status()

            while deploy_status == DeployStatus.DEPLOYING.value:
                # Reset attempts
                tsum_pipeline_retry = 5
                deploy_status = self._get_mt_status()

                time.sleep(30)

            tsum_pipeline_retry -= 1
            time.sleep(15)

        if deploy_status != DeployStatus.READY.value:
            raise Exception("MT pipeline failed")

    def _get_mt_status(self):
        headers = {
            "Authorization": self._robot_secret.data()['oauth-token-tsum'],
            "Content-Type": "application/json"
        }

        retry_count = 5

        self.Parameters.mt_url = mt_utils.balancer_url(self.Parameters.tsum_project, self._tsum_mt_name)

        while retry_count > 0:
            status = send_request("get", self._tsum_mt_status, headers=headers)
            if status.status_code > 299:
                # If service unavailable wait 1 minute and retry
                retry_count -= 1
                logging.error(u"Bad TSUM response: {}".format(status.text))
                time.sleep(20)
            else:
                return status.json()['status']

        raise Exception("Too many requests errors")

    def on_execute(self):
        super(WmsWaitMTAutotests, self).on_execute()
        self.wait_for_deploy()
