# -*- coding: utf-8 -*-

"""
@author: evgeny-bond@yandex-team.ru
Модуль сборки настроек фича тоггла для маркета
Подробная информация: https://st.yandex-team.ru/MARKETFRONTECH-1627
"""

import logging
from sandbox import sdk2
import sandbox.sdk2.path as spath

from sandbox.sdk2.vcs.git import Git
from sandbox.projects.market.front.helpers.github import build_repo_url
import os.path
import pipes
import tempfile
import sandbox.common.types.misc as ctm
from sandbox.projects.common.nanny import nanny

from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv

from sandbox.projects.market.resources import MARKET_WRATH


class updateMarketWrath(nanny.ReleaseToNannyTask2, sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Приложение") as app:
            app_owner = sdk2.parameters.String(
                "Github owner",
                default_value="market",
            )

            app_repo = sdk2.parameters.String(
                "Github repo",
                default_value="wrath",
                required=True
            )
            app_branch = sdk2.parameters.String(
                "Ветка",
                default_value="master",
                required=True
            )
            app_revision = sdk2.parameters.String(
                "Git revision",
                required=False,
            )
            ubuntu_version = create_ubuntu_selector()
            node_version = create_node_selector()

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    def on_enqueue(self):
        super(updateMarketWrath, self).on_enqueue()
        setup_container(self)

    def on_execute(self):
        super(updateMarketWrath, self).on_execute()

        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            root_dir = tempfile.mkdtemp()
            app_src_path = os.path.join(root_dir, self.Parameters.app_repo)

            self._clone_repo(app_src_path)
            self._build_config(app_src_path)
            self._validate_config(app_src_path)
            self._create_config_pack(app_src_path)

    def _clone_repo(self, app_src_path):
        url = build_repo_url(self.Parameters.app_owner, self.Parameters.app_repo)
        git = Git(url)
        logging.info("Checking out sources from {}".format(url))
        git.clone(str(app_src_path), pipes.quote(self.Parameters.app_branch), commit=self.Parameters.app_revision)

    def _build_config(self, app_src_path):
        logging.info("Installing dependencies")
        rich_check_call(
            ["npm", "ci"],
            task=self, alias="npm_install", cwd=app_src_path
        )
        logging.info("Running build")
        rich_check_call(
            ["npm", "run", "build"],
            task=self, alias="npm_build", cwd=app_src_path
        )

    def _validate_config(self, app_src_path):
        logging.info("Running validate")
        rich_check_call(
            ["npm", "run", "validate"],
            task=self, alias="npm_validate", cwd=app_src_path
        )

    def _create_config_pack(self, app_src_path):
        logging.info("Creating resource from new config")
        pack_app_archive_path = tempfile.mktemp(suffix=".tar.gz", prefix=self.Parameters.app_repo)

        rich_check_call(
            ["tar", "-C", os.path.join(app_src_path, 'dist'), "-czf", pack_app_archive_path, "wrath.config.json"],
            task=self, alias="create_app_pack"
        )

        self._app_create_resource(pack_app_archive_path)

    def _app_create_resource(self, pack_app_archive_path):
        app_res = sdk2.ResourceData(MARKET_WRATH(
            self, "Wrath aka Market feature toggle config", "{}.tar.gz".format(self.Parameters.app_repo)
        ))

        app_res.path.write_bytes(spath.Path(pack_app_archive_path).read_bytes())
        app_res.ready()
