# -*- coding: utf-8 -*-
from sandbox import sdk2
import sandbox.common.types.task as ctt


class Yt2Binfile(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        yt_proxy = sdk2.parameters.String('YT cluster', required=True)
        yt_token = sdk2.parameters.YavSecret("YT token", default="sec-01fyqb996a92vxpgx8n8ptgy55")
        input_table = sdk2.parameters.String('Table to dump to file', required=True)
        resource_name = sdk2.parameters.String('Access resource name', required=True)

        access_endpoint = sdk2.parameters.String(
            'Endpoint to publish', description='Values: prod, test', required=True, default_value='prod'
        )

        indices = sdk2.parameters.List("Индексы. Формат: index_name:key_column")
        formatted_indices = sdk2.parameters.List(
            "Индексы задаваемые форматной fmt строкой. Можно использовать имена колонок как переменные. Формат: index_name:index_fmt_string."
            " Пример: index:{hid}_{name}. На выходе получится один индекс с составленым из 2 колонок ключем"
        )
        schema_path = sdk2.parameters.String('Arcadia schema path', required=True)
        data_prefix = sdk2.parameters.String('Data prefix', required=True)
        columns = sdk2.parameters.List("Column(s) to get from table", required=True)
        use_flatbuf64 = sdk2.parameters.Bool("Use flatbuf64")

    def on_create(self):
        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={"Name": "Yt2Binfile", "release": ctt.ReleaseStatus.STABLE},
        ).first()

    def on_execute(self):
        import subprocess
        import os
        from sandbox.projects.common import link_builder as lb
        from sandbox.projects.common.arcadia import sdk as arcadia_sdk
        from sandbox.projects.market.yt2binfile.resources import Yt2BinfileResource
        from sandbox.projects.market.report.PublishMarketAccessResource import PublishMarketAccessResource

        with self.memoize_stage.prepare_data:
            binary_resource = sdk2.Resource.find(type="YT_2_BINFILE_BINARY", attrs={"released": "stable"}).first()

            data = sdk2.ResourceData(binary_resource)
            yt2binfile_path = data.path

            self.set_info("Using yt2binfile from resource {} path {}".format(binary_resource.id, yt2binfile_path))

            with arcadia_sdk.mount_arc_path("arcadia-arc:/#trunk") as arc_path:
                schema_path = os.path.join(arc_path, self.Parameters.schema_path)
                with open(schema_path) as schema_fn:
                    self.set_info("Schema {}:\n{}".format(self.Parameters.schema_path, schema_fn.read()))

                yt_token = self.Parameters.yt_token.data()["yt_token"]
                env = os.environ.copy()
                env["YT_TOKEN"] = yt_token

                resource = Yt2BinfileResource(self, "yt2binfile resource", self.Parameters.resource_name)
                resource_data = sdk2.ResourceData(resource)
                resource_data.path.mkdir(parents=True, exist_ok=True)

                dst_dir = str(resource_data.path)

                cmd = [
                    str(yt2binfile_path),
                    '--yt-proxy',
                    self.Parameters.yt_proxy,
                    '--input-table',
                    self.Parameters.input_table,
                    '--resource-name',
                    self.Parameters.resource_name,
                    '--dst-dir',
                    dst_dir,
                    '--schema-path',
                    schema_path,
                    '--data-prefix',
                    self.Parameters.data_prefix,
                ]

                for column in self.Parameters.columns:
                    cmd.append('--column')
                    cmd.append(column)

                for index in self.Parameters.indices:
                    index_name, index_key = index.split(':', 1)

                    cmd.append('--index-name')
                    cmd.append(index_name)

                    cmd.append('--key')
                    cmd.append(index_key)

                for index in self.Parameters.formatted_indices:
                    index_name, index_key = index.split(':', 1)

                    cmd.append('--formatted-index-name')
                    cmd.append(index_name)

                    cmd.append('--formatted-key')
                    cmd.append(index_key)

                if self.Parameters.use_flatbuf64:
                    cmd.append('--use-flat64')

                cmd = map(str, cmd)

                self.set_info("will run:{}".format(subprocess.list2cmdline(cmd)))

                with sdk2.helpers.ProcessLog(self, logger="shell") as pl:
                    sdk2.helpers.subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr, env=env)

                self.set_info("Files done:{}".format(os.listdir(dst_dir)))

                resource_data.ready()
                self.set_info("Resource {} done".format(lb.resource_link(resource.id)), do_escape=False)

                if self.Parameters.resource_name and self.Parameters.access_endpoint:
                    publish_access_task = PublishMarketAccessResource(
                        self,
                        sandbox_resource=resource.id,
                        resource_name=self.Parameters.resource_name,
                        endpoint=self.Parameters.access_endpoint,
                    )

                    self.set_info(
                        'Publish resource to access task {} ...'.format(lb.task_link(publish_access_task.id)),
                        do_escape=False,
                    )
                    publish_access_task.enqueue()
                    raise sdk2.WaitTask([publish_access_task], ctt.Status.Group.FINISH, wait_all=True)

        self.set_info("All DONE!")
