import binascii
import logging
import os
import subprocess
import random
import json
from datetime import datetime

import sandbox.common.platform
import sandbox.common.types.resource
import sandbox.common.urls
import sandbox.sdk2

_YT_CLUSTER = 'hahn'
_YT_TOKEN_KEY = "robot_robot-mtr_yt_token"
_THREE_MONTHS = 90 * 24 * 60 * 60 * 1000

_TVMTOOL_PORT = 12346
_TVMTOOL_CACHE_DIR = "tvmtool-cache-dir"
_YAV_SECRET_ID = 'sec-01f5jyk087p28t19ydezkqezx2'

_ENVIRONMENT_CONFIGS = {}

DATE_STRING_COLUMN_NAME = "date_string"
TOTAL_ROUTES_COLUMN_NAME = "total_routes"
COMPARED_ROUTES_COLUMN_NAME = "compared_routes"
FAILED_REQUESTS_COLUMN_NAME = "failed_requests"
FIRST_ROUTER_WINS_COLUMN_NAME = "first_wins"
SECOND_ROUTER_WINS_COLUMN_NAME = "second_wins"
SRCS_SIZE_COLUMN_NAME = "srcs_size"
DSTS_SIZE_COLUMN_NAME = "dsts_size"
MIN_SIGNED_DIFF_COLUMN_NAME = "min_signed_diff"
MAX_SIGNED_DIFF_COLUMN_NAME = "max_column_name"
AVG_SIGNED_DIFF_COLUMN_NAME = "avg_signed_diff"
MIN_ABS_DIFF_COLUMN_NAME = "min_abs_diff"
MAX_ABS_DIFF_COLUMN_NAME = "max_abs_diff"
AVG_ABS_DIFF_COLUMN_NAME = "avg_abs_diff"
MEDIAN_ABS_DIFF_COLUMN_NAME = "median_abs_diff"
MIN_RELATIVE_DIFF_COLUMN_NAME = "min_relative_diff"
MAX_RELATIVE_DIFF_COLUMN_NAME = "max_relative_diff"
AVG_RELATIVE_DIFF_COLUMN_NAME = "avg_relative_diff"
MEDIAN_RELATIVE_DIFF_COLUMN_NAME = "median_relative_diff"
REQUEST_COLUMN_NAME = "request"
SRC_IDX_COLUMN_NAME = "src_idx"
DST_IDX_COLUMN_NAME = "dst_idx"
YMAPSDF_TIME_COLUMN_NAME = "ymapsdf_time"
OSM_TIME_COLUMN_NAME = "osm_time"
YMAPSDF_DISTANCE_COLUMN_NAME = "ymapsdf_distance"
OSM_DISTANCE_COLUMN_NAME = "osm_distance"
YMAPSDF_RESULT_COLUMN_NAME_PREFIX = "ymapsdf_"
OSM_RESULT_COLUMN_NAME_PREFIX = "osm_"


def _get_tvm_config(ymapsdf_service_dst_id, osm_service_dst_id):
    config_format = """{
    "clients": {
        "maps-core-masstransit-testing-tools": {
            "secret": "env:TVM_SECRET",
            "self_tvm_id": 2028046,
            "dsts": {
                "maps-core-bicycle-matrix": {
                    "dst_id": """ + str(ymapsdf_service_dst_id) + """
                },
                "maps-core-bicycle-matrix-over-osm": {
                    "dst_id": """ + str(osm_service_dst_id) + """
                }
            }
        }
    }
}
"""
    return config_format


def _random_auth_token():
    """Generate a random string to be used as a TVM authorization token.

    More on this token:
    https://wiki.yandex-team.ru/passport/tvm2/tvm-daemon/#kakzapustitprocess
    """
    return binascii.hexlify(os.urandom(16)).decode("utf-8")


def _prepare_tvmtool():
    platform = sandbox.common.platform.platform()
    arch = sandbox.common.platform.get_arch_from_platform(platform)
    tvmtool_resource = sandbox.sdk2.Resource.find(
        type="TVM_TOOL_BINARY", arch=arch).first()
    return str(sandbox.sdk2.ResourceData(tvmtool_resource).path)


def _dump_table_schema():
    schema = {"schema": [
        {"name": REQUEST_COLUMN_NAME, "type_v3": "string"},
        {"name": SRC_IDX_COLUMN_NAME, "type_v3": "int64"},
        {"name": DST_IDX_COLUMN_NAME, "type_v3": "int64"},
        {"name": YMAPSDF_TIME_COLUMN_NAME, "type_v3": "double"},
        {"name": OSM_TIME_COLUMN_NAME, "type_v3": "double"},
        {"name": YMAPSDF_DISTANCE_COLUMN_NAME, "type_v3": "double"},
        {"name": OSM_DISTANCE_COLUMN_NAME, "type_v3": "double"}
    ]}
    return schema


class MapsBicycleCompareMatrixRouters(sandbox.sdk2.Task):

    class Requirements(sandbox.sdk2.Requirements):
        cores = 1
        disk_space = 1024  # MB
        ram = 150 * 1024  # MB

    class Parameters(sandbox.sdk2.Task.Parameters):
        kill_timeout = 15 * 60  # seconds

        with sandbox.sdk2.parameters.RadioGroup("Region") as region:
            region.values.moscow = region.Value(
                value="moscow", default=True)

        with sandbox.sdk2.parameters.RadioGroup("Environment") as environment:
            environment.values.testing = environment.Value(
                value="testing")
            environment.values.datatesting = environment.Value(
                value="datatesting")
            environment.values.stable = environment.Value(
                value="stable", default=True)

        num_of_requests = sandbox.sdk2.parameters.Integer(
            "Number of requests to shoot", default=1000)
        save_responses_dump = sandbox.sdk2.parameters.Bool(
            "Dump requests/responses to file", default=False)
        yt_token_secret_name = sandbox.sdk2.parameters.String(
            "Sandbox vault secret name for YT token", default=_YT_TOKEN_KEY)

    def _tvm_secret(self):
        yav_secret = sandbox.sdk2.yav.Secret(_YAV_SECRET_ID)
        return yav_secret.data()["client_secret"]

    def on_create(self):
        if self.Requirements.tasks_resource is None:
            # This code does not seem to work when running a task locally with
            # --enable-taskbox. The if-else solves this.
            logging.info(
                "task binary is not set, looking for latest suitable one")
            self.Requirements.tasks_resource = (
                sandbox.sdk2.service_resources.SandboxTasksBinary.find(
                    attrs={"target": "maps/MapsBicycleCompareMatrixRouters"},
                    state=sandbox.common.types.resource.State.READY
                ).first()
            )
            logging.info("task binary: {}".format(
                self.Requirements.tasks_resource))
        else:
            logging.info("task binary is already set")

    def _write_dump_to_yt(self, compare_result, table_name):
        import yt.wrapper as yt
        yt.config.set_proxy(_YT_CLUSTER)
        yt.config['token'] = sandbox.sdk2.Vault.data(self.Parameters.yt_token_secret_name)
        with yt.Transaction(timeout=5*60*1000):  # 5 minutes
            output_table_path = _ENVIRONMENT_CONFIGS[self.Parameters.environment]["dump_results_yt_folder"] + table_name
            yt_table_directory = yt.ypath_dirname(output_table_path)
            if not yt.exists(yt_table_directory):
                yt.mkdir(yt_table_directory)
            yt.create_table(
                output_table_path,
                attributes=_dump_table_schema())
            yt.set_attribute(output_table_path, "expiration_timeout", _THREE_MONTHS)

            rows = []
            for shooting_result in compare_result.results:
                for i in range(compare_result.srcs_size):
                    for j in range(compare_result.dsts_size):
                        row = {
                            REQUEST_COLUMN_NAME: shooting_result["request"],
                            SRC_IDX_COLUMN_NAME: i,
                            DST_IDX_COLUMN_NAME: j,
                            YMAPSDF_TIME_COLUMN_NAME: shooting_result["first_values"][i * compare_result.srcs_size + j]["time"],
                            OSM_TIME_COLUMN_NAME: shooting_result["second_values"][i * compare_result.srcs_size + j]["time"],
                            YMAPSDF_DISTANCE_COLUMN_NAME: shooting_result["first_values"][i * compare_result.srcs_size + j]["distance"],
                            OSM_DISTANCE_COLUMN_NAME: shooting_result["second_values"][i * compare_result.srcs_size + j]["distance"]
                        }
                        rows.append(row)
            yt.write_table(yt.TablePath(
                output_table_path),
                rows)
            logging.info("Results saved on YT at {}".format(output_table_path))

    def _write_to_yt(self, compare_result):
        current_date_str = datetime.utcnow().strftime("%Y-%m-%d_%H:%M:%S")
        import yt.wrapper as yt
        yt.config.set_proxy(_YT_CLUSTER)
        yt.config['token'] = sandbox.sdk2.Vault.data(self.Parameters.yt_token_secret_name)
        with yt.Transaction(timeout=5*60*1000):  # 5 minutes
            output_table = _ENVIRONMENT_CONFIGS[self.Parameters.environment]["yt_output_table"]
            if not yt.exists(output_table):
                yt_table_directory = yt.ypath_dirname(output_table)
                if not yt.exists(yt_table_directory):
                    yt.mkdir(yt_table_directory)
                yt.create_table(
                    output_table,
                    attributes={"schema": [
                        {"name": DATE_STRING_COLUMN_NAME,
                            "type": "string", "sort_order": "ascending"},
                        {"name": TOTAL_ROUTES_COLUMN_NAME, "type": "int64"},
                        {"name": COMPARED_ROUTES_COLUMN_NAME, "type": "int64"},
                        {"name": FAILED_REQUESTS_COLUMN_NAME, "type": "int64"},
                        {"name": FIRST_ROUTER_WINS_COLUMN_NAME, "type": "int64"},
                        {"name": SECOND_ROUTER_WINS_COLUMN_NAME, "type": "int64"},
                        {"name": SRCS_SIZE_COLUMN_NAME, "type": "int64"},
                        {"name": DSTS_SIZE_COLUMN_NAME, "type": "int64"},
                        {"name": MIN_SIGNED_DIFF_COLUMN_NAME, "type": "double"},
                        {"name": MAX_SIGNED_DIFF_COLUMN_NAME, "type": "double"},
                        {"name": AVG_SIGNED_DIFF_COLUMN_NAME, "type": "double"},
                        {"name": MIN_ABS_DIFF_COLUMN_NAME, "type": "double"},
                        {"name": MAX_ABS_DIFF_COLUMN_NAME, "type": "double"},
                        {"name": AVG_ABS_DIFF_COLUMN_NAME, "type": "double"},
                        {"name": MEDIAN_ABS_DIFF_COLUMN_NAME, "type": "double"},
                        {"name": MIN_RELATIVE_DIFF_COLUMN_NAME, "type": "double"},
                        {"name": MAX_RELATIVE_DIFF_COLUMN_NAME, "type": "double"},
                        {"name": AVG_RELATIVE_DIFF_COLUMN_NAME, "type": "double"},
                        {"name": MEDIAN_RELATIVE_DIFF_COLUMN_NAME, "type": "double"}
                    ]}
                )

            row = {
                DATE_STRING_COLUMN_NAME: current_date_str,
                TOTAL_ROUTES_COLUMN_NAME: compare_result.total_routes,
                COMPARED_ROUTES_COLUMN_NAME: compare_result.compared_routes,
                FAILED_REQUESTS_COLUMN_NAME: compare_result.failed_requests,
                FIRST_ROUTER_WINS_COLUMN_NAME: compare_result.first_wins,
                SECOND_ROUTER_WINS_COLUMN_NAME: compare_result.second_wins,
                SRCS_SIZE_COLUMN_NAME: compare_result.srcs_size,
                DSTS_SIZE_COLUMN_NAME: compare_result.dsts_size,
                MIN_SIGNED_DIFF_COLUMN_NAME: compare_result.min_signed_diff,
                MAX_SIGNED_DIFF_COLUMN_NAME: compare_result.max_signed_diff,
                AVG_SIGNED_DIFF_COLUMN_NAME: compare_result.avg_signed_diff,
                MIN_ABS_DIFF_COLUMN_NAME: compare_result.min_abs_diff,
                MAX_ABS_DIFF_COLUMN_NAME: compare_result.max_abs_diff,
                AVG_ABS_DIFF_COLUMN_NAME: compare_result.avg_abs_diff,
                MEDIAN_ABS_DIFF_COLUMN_NAME: compare_result.median_abs_diff,
                MIN_RELATIVE_DIFF_COLUMN_NAME: compare_result.min_relative_diff,
                MAX_RELATIVE_DIFF_COLUMN_NAME: compare_result.max_relative_diff,
                AVG_RELATIVE_DIFF_COLUMN_NAME: compare_result.avg_relative_diff,
                MEDIAN_RELATIVE_DIFF_COLUMN_NAME: compare_result.median_relative_diff
            }
            yt.write_table(yt.TablePath(
                output_table,
                append=True),
                [row])
            logging.info("Statistics added on YT at {}".format(output_table))

        if self.Parameters.save_responses_dump:
            self._write_dump_to_yt(compare_result, current_date_str)

    def on_execute(self):
        from library.python import resource
        env_config = resource.find("/environments_config.json").decode('ascii')
        global _ENVIRONMENT_CONFIGS
        _ENVIRONMENT_CONFIGS = json.loads(env_config)
        logging.info("checking that tasks resource is properly set")
        logging.info("current tasks_resource: {}".format(
            self.Requirements.tasks_resource))
        if self.Requirements.tasks_resource is None:
            raise sandbox.common.errors.TaskFailure(
                "self.Requirements.tasks_resource is not set"
                " for MapsBicycleCompareMatrixRouters")

        tvm_config_path = "tvm-config.json"
        with open(tvm_config_path, "w") as tvm_config:
            tvm_config.write(_get_tvm_config(
                _ENVIRONMENT_CONFIGS[self.Parameters.environment]["ymapsdf_tvm_id"],
                _ENVIRONMENT_CONFIGS[self.Parameters.environment]["osm_tvm_id"]))

        os.environ["TVMTOOL_LOCAL_AUTHTOKEN"] = _random_auth_token()
        os.environ["TVM_SECRET"] = self._tvm_secret()
        os.makedirs(_TVMTOOL_CACHE_DIR)

        tvmtool_path = _prepare_tvmtool()
        with sandbox.sdk2.helpers.ProcessLog(self, logger="tvmtool") as tvm_log:
            tvmtool_process = subprocess.Popen(
                [
                    tvmtool_path,
                    "-c", tvm_config_path,
                    "--port", str(_TVMTOOL_PORT),
                    "--cache-dir", _TVMTOOL_CACHE_DIR,
                ],
                stdout=tvm_log.stdout,
                stderr=tvm_log.stderr)
            logging.info("shooting bicycle matrix router in testing")
            random.seed(42)

            from maps.bicycle.tools.matrix_compare.lib.routers_comparator import RoutersComparator, REGIONS

            comparator = RoutersComparator(
                first_router_url=_ENVIRONMENT_CONFIGS[self.Parameters.environment]["ymapsdf_router_url"],
                second_router_url=_ENVIRONMENT_CONFIGS[self.Parameters.environment]["osm_router_url"],
                num_of_requests=self.Parameters.num_of_requests,
                bounding_box=REGIONS[self.Parameters.region],
                tvm_source_alias="maps-core-masstransit-testing-tools",
                tvm_first_dst_alias="maps-core-bicycle-matrix",
                tvm_second_dst_alias="maps-core-bicycle-matrix-over-osm",
                tvm_auth_token=None,
                use_tvm=True,
                tvmtool_port=_TVMTOOL_PORT,
                dump_results=self.Parameters.save_responses_dump
            )

            self._write_to_yt(comparator.compare())

            tvmtool_process.terminate()
