from sandbox import sdk2
from sandbox.projects.masstransit.common.shooting import make_simple_load_config, make_step_rps
from sandbox.projects.masstransit.common.subtasks import ParentTask, SubtasksError
from sandbox.projects.masstransit.common.utils import make_shooting_stage
from sandbox.projects.masstransit.MapsMasstransitPrepareProductionLogAmmo import MapsMasstransitPrepareProductionLogAmmo
from sandbox.projects.masstransit.MapsMasstransitRestartService import MapsMasstransitRestartService
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi

AMMO_RESOURCE_TYPE='MAPS_MASSTRANSIT_PRODUCTION_LOG_AMMO'


class MapsBicycleMatrixShooting(ParentTask):
    '''
    Bicycle matrix regression load testing
    '''

    class Parameters(sdk2.Parameters):
        shooting_task = sdk2.parameters.String('Shooting task', default='MTDEV-987')
        shoot_with_generated_ammo = sdk2.parameters.Bool('Run additional load test with generated ammo first', default=False)
        with sdk2.parameters.Group('Target service', collapse=False) as target_service:
            service_vhost = sdk2.parameters.String('Service vhost', default='core-bicycle-matrix.maps.yandex.net', required=True)
            nanny_service = sdk2.parameters.String('Nanny service', default='maps_core_bicycle_matrix_load', required=True)
            yav_secret = sdk2.parameters.String('Target service secret', default='sec-01f767b2vw8cx1rt09q0tadh7x', required=True)  # TVM secret for bicycle-matrix-testing
            telegraf_config = sdk2.parameters.String('Telegraf config', default='maps/bicycle/matrix/regression/telegraf_config.cfg', required=True)
            restart_service = sdk2.parameters.Bool('Restart service', default=True)
            service_short_name = sdk2.parameters.String('Service short name', default='bicycle_matrix', required=True)

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def generate_bicycle_matrix_ammo(self, handle, requests_number=1000000):
        stage = 'Ammo generator for {0} ({1})'.format(handle, self.Parameters.service_short_name)
        ammo_task = self.run_subtask(
            stage,
            lambda: MapsMasstransitPrepareProductionLogAmmo(
                self,
                description=stage,
                notifications=self.Parameters.notifications,
                resource_type=AMMO_RESOURCE_TYPE,
                vhost=self.Parameters.service_vhost,
                requests_number=requests_number,
                handles=[handle]
            )
        )
        return ammo_task.Parameters.ammo

    def shoot_bicycle_matrix(self, handle, rps, ammo_resource, autostop, shooting_name_suffix, post_ammo=False, anonymous=False):
        headers = [
            'Host: {0}'.format(self.Parameters.service_vhost),
            'Accept: application/x-protobuf'
        ]

        if not anonymous:
            yav_secret = sdk2.yav.Secret(self.Parameters.yav_secret)
            headers += [
                'X-Ya-Service-Ticket: {0}'.format(yav_secret.data()["client_secret"])
            ]

        config_content = make_simple_load_config(
            task=self.Parameters.shooting_task,
            component=handle,
            rps=rps,
            headers=headers,
            autostop=autostop,
            shooting_name_suffix=shooting_name_suffix,
            ammo_type='uripost' if post_ammo else 'uri'
        )

        stage = make_shooting_stage(handle, rps, shooting_name_suffix)

        self.run_subtask(
            stage,
            lambda: ShootViaTankapi(
                self,
                description='Shooting {0} with rps {1}'.format(handle, rps),
                notifications=self.Parameters.notifications,
                use_public_tanks=False,
                tanks=['nanny:maps_core_tanks_load'],
                nanny_service=self.Parameters.nanny_service,
                config_source='file',
                config_content=config_content,
                use_monitoring=True,
                monitoring_source='arcadia',
                monitoring_arc_path=self.Parameters.telegraf_config,
                ammo_source='resource',
                ammo_resource=ammo_resource
            )
        )

    def run_load_test(self, handle, bicycle_matrix_rps, ammo_resource, autostop=[],
                      shooting_name_suffix=None, post_ammo=False):
        try:
            for rps in bicycle_matrix_rps:
                try:
                    self.shoot_bicycle_matrix(
                        handle,
                        rps,
                        ammo_resource,
                        autostop,
                        shooting_name_suffix,
                        post_ammo,
                        anonymous=True
                    )
                except SubtasksError:
                    pass
        except SubtasksError:
            # Do not fail the parent task, continue with next load tests
            pass

    def restart_bicycle_matrix(self):
        self.run_subtask(
            'Restart {0}'.format(self.Parameters.service_short_name),
            lambda: MapsMasstransitRestartService(
                self,
                description='Restarting {0} before load tests'.format(self.Parameters.service_short_name),
                nanny_service_id=self.Parameters.nanny_service
            )
        )

    def on_execute(self):
        if self.Parameters.restart_service:
            self.restart_bicycle_matrix()

        handle = '/v2/matrix'
        if self.Parameters.shoot_with_generated_ammo:
            self.run_load_test(
                handle=handle,
                bicycle_matrix_rps=[
                    make_step_rps(max_rps=250)
                ],
                ammo_resource=self.generate_bicycle_matrix_ammo(handle),
                shooting_name_suffix=' ({0})'.format(self.Parameters.service_short_name)
            )

        self.run_load_test(
            handle=handle,
            bicycle_matrix_rps=[
                make_step_rps(max_rps=140)
            ],
            ammo_resource='2256842971',
            shooting_name_suffix='random matrices 10x10 ({0})'.format(self.Parameters.service_short_name)
        )
