from sandbox import sdk2
from sandbox.projects.masstransit.common.shooting import Target, make_simple_load_config, make_step_rps
from sandbox.projects.masstransit.common.subtasks import ParentTask, SubtasksError
from sandbox.projects.masstransit.common.utils import make_shooting_stage
from sandbox.projects.masstransit.MapsMasstransitPrepareProductionLogAmmo import MapsMasstransitPrepareProductionLogAmmo
from sandbox.projects.masstransit.MapsMasstransitRestartService import MapsMasstransitRestartService
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi


class MapsBicycleRouterShooting(ParentTask):

    class Parameters(sdk2.Parameters):
        restart_service = sdk2.parameters.Bool('Restart service', default=True)

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass


    def generate_bicycle_router_ammo(self, handle, requests_number=1000000):
        stage = 'Ammo generator for {0}'.format(handle)
        ammo_task = self.run_subtask(
            stage,
            lambda: MapsMasstransitPrepareProductionLogAmmo(
                self,
                description=stage,
                notifications=self.Parameters.notifications,
                resource_type='MAPS_MASSTRANSIT_PRODUCTION_LOG_AMMO',
                vhost='core-bicycle-router.maps.yandex.net',
                requests_number=requests_number,
                handles=[handle]
            )
        )
        return ammo_task.Parameters.ammo

    def shoot_bicycle_router(self, handle, rps, ammo_resource, name_suffix):
        headers = [
            'Host: core-bicycle-router.maps.yandex.net',
            'Accept: application/x-protobuf',
            'X-Ya-Service-Ticket: {0}'.format(sdk2.Vault.data('maps-core-bicycle-router-load-tvm-ticket')),
        ]
        config_content = make_simple_load_config(
            task='MTDEV-177',
            component=handle,
            rps=rps,
            headers=headers,
            shooting_name_suffix=name_suffix
        )
        stage = make_shooting_stage(handle, rps, shooting_name_suffix=name_suffix)

        self.run_subtask(
            stage,
            lambda: ShootViaTankapi(
                self,
                description='Shooting {0} with rps {1}'.format(handle, rps),
                notifications=self.Parameters.notifications,
                use_public_tanks=False,
                tanks=['nanny:maps_core_tanks_load'],
                nanny_service='maps_core_bicycle_router_load',
                config_source='file',
                config_content=config_content,
                use_monitoring=True,
                monitoring_source='arcadia',
                monitoring_arc_path='maps/bicycle/router/regression/telegraf_config.cfg',
                ammo_source='resource',
                ammo_resource=ammo_resource
            )
        )

    def run_load_test(self, handle, bicycle_router_rps, ammo=None, name_suffix=None):
        try:
            if ammo is None:
                ammo = self.generate_bicycle_router_ammo(handle)
            for rps in bicycle_router_rps:
                self.shoot_bicycle_router(handle, rps, ammo, name_suffix)
        except SubtasksError:
            # Do not fail the parent task, continue with next load tests
            pass


    def restart_router(self):
        self.run_subtask(
            'Restart BicycleRouter',
            lambda: MapsMasstransitRestartService(
                self,
                description='Restarting BicycleRouter before load tests',
                nanny_service_id='maps_core_bicycle_router_load'
            )
        )


    def on_execute(self):
        if self.Parameters.restart_service:
            self.restart_router()

        self.run_load_test(
            handle='/v2/route',
            bicycle_router_rps=[
                make_step_rps(700)
            ]
        )

        self.run_load_test(
            handle='/v2/summary',
            bicycle_router_rps=[
                make_step_rps(3400)
            ]
        )

        self.run_load_test(
            handle='/v2/uri',
            bicycle_router_rps=[
                make_step_rps(1000)
            ]
        )

        self.run_load_test(
            handle='/v2/matrix',
            bicycle_router_rps=[
                make_step_rps(1200)
            ],
            ammo='1595873913',
            name_suffix='random matrices 1x25'
        )
        self.run_load_test(
            handle='/v2/matrix',
            bicycle_router_rps=[
                make_step_rps(900)
            ],
            ammo='1595913094',
            name_suffix='random matrices 1x50'
        )
        self.run_load_test(
            handle='/v2/matrix',
            bicycle_router_rps=[
                make_step_rps(300)
            ],
            ammo='1595955712',
            name_suffix='random matrices 1x100'
        )
