from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.projects import resource_types
from sandbox.projects.geobase.Geodata5BinStable.resource import GEODATA5BIN_STABLE
from sandbox.projects.masstransit.resources import MASSTRANSIT_DATA
from sandbox.projects.masstransit.common.utils import (
    get_latest_stable_resource, get_stable_masstransit_data_resource, unpack_tzdata
)

import logging
import os


DEFAULT_GRAPH_PATH_TEMPLATE = "//home/maps/core/garden/stable/pedestrian_graph_fb/{version}"


class MapsMasstransitBuildSnippetsToolBinary(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ["MAPS-MASSTRANSIT"]


def get_default_graph_path(version):
    return DEFAULT_GRAPH_PATH_TEMPLATE.format(version=version)


class MapsMasstransitBuildSnippets(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        disk_space = 10 * 1024  # 10 gb for masstransit-data
        ram = 15 * 1024  # Again, for masstransit data.
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 5 * 3600

        build_snippets_bin = sdk2.parameters.Resource(
            "Build snippets tool resource",
            description="Resource with the tool to build snippets. "
            "If not specified, latest stable release will be used.",
            resource_type=MapsMasstransitBuildSnippetsToolBinary
        )
        masstransit_data = sdk2.parameters.Resource(
            "Masstransit-data resource",
            description="Resource with masstransit-data dataset from garden. "
            "If not specified, latest stable release will be used.",
            resource_type=MASSTRANSIT_DATA
        )
        pedestrian_graph_path = sdk2.parameters.String(
            "Pedestrian graph path",
            description="Path to pedestrian graph on Hahn. "
            "If not specified, version from the stable garden corresponding "
            "to the masstransit-data release will be used."
        )
        output_dir = sdk2.parameters.String(
            "Output folder",
            description="Path to the output folder on Hahn.",
            required=True
        )
        geodata = sdk2.parameters.Resource(
            "Resource with geodata5.bin",
            resource_type=GEODATA5BIN_STABLE
        )
        tzdata = sdk2.parameters.Resource(
            "Resource with geobase tzdata",
            resource_type=resource_types.GEODATATZDATA_STABLE
        )

    def on_save(self):
        # TODO: Reuse common.utils.LatestResourceParameter here.
        if not self.Parameters.build_snippets_bin:
            self.Parameters.build_snippets_bin = get_latest_stable_resource(
                MapsMasstransitBuildSnippetsToolBinary)
        if not self.Parameters.masstransit_data:
            self.Parameters.masstransit_data = get_stable_masstransit_data_resource()
        if not self.Parameters.pedestrian_graph_path:
            self.Parameters.pedestrian_graph_path = get_default_graph_path(
                self.Parameters.masstransit_data.version)
        if not self.Parameters.geodata:
            self.Parameters.geodata = get_latest_stable_resource(GEODATA5BIN_STABLE)
        if not self.Parameters.tzdata:
            self.Parameters.tzdata = get_latest_stable_resource(resource_types.GEODATATZDATA_STABLE)

    def on_execute(self):
        tool = sdk2.ResourceData(self.Parameters.build_snippets_bin)
        masstransit_data = sdk2.ResourceData(self.Parameters.masstransit_data)
        geodata = sdk2.ResourceData(self.Parameters.geodata)
        tzdata_archive = sdk2.ResourceData(self.Parameters.tzdata)
        unpacked_tzdata = unpack_tzdata(str(tzdata_archive.path))

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("build-snippets")) as process_log:
            command = [
                str(tool.path),
                "--geodata", str(geodata.path),
                "--masstransit-data", str(masstransit_data.path / "static.fb"),
                "--graph", str(self.Parameters.pedestrian_graph_path),
                "--working-dir", str(self.Parameters.output_dir)
            ]
            subprocess.check_call(
                command,
                stdout=process_log.stdout,
                stderr=subprocess.STDOUT,
                env={
                    "YT_TOKEN": sdk2.Vault.data("robot_robot-mtr_yt_token"),
                    "masstransit-geobase.geodataPath": str(geodata.path),
                    "masstransit-geobase.tzdataPath": os.path.join(unpacked_tzdata, "zones_bin"),
                }
            )
