import csv
import logging
import math
import os
from itertools import groupby, imap

from sandbox import sdk2
from sandbox.sandboxsdk import errors, paths


YT_CLUSTER = "hahn"
YQL_TOKEN_KEY = "robot_robot-mtr_yql_token"
YT_TOKEN_KEY = "robot_robot-mtr_yt_token"
SCRIPT_TEMPLATE_KEY = "sandbox/projects/masstransit/MapsMasstransitCurrentRoutesReport/query.sql"
INDEX_HTML_TEMPLATE_KEY = "sandbox/projects/masstransit/MapsMasstransitCurrentRoutesReport/index.html"
LOGS_PATH = "//logs/analyzer-dispatcher-signals-log/stream/5min"
LOGS_SPAN_MINUTES = 5


def ceil_div(x, y):
    return int(math.ceil(1. * x / y))


def _write_report(html_template, path, clids):
    from jinja2.environment import Template

    stream = Template(html_template, trim_blocks=True, lstrip_blocks=True)\
        .stream(clids=clids)
    with open(os.path.join(path, "index.html"), "w") as f:
        stream.dump(f)


def _write_csv(data, fieldnames, path):
    with open(path, "w") as file:
        writer = csv.writer(file)
        writer.writerow(fieldnames)
        writer.writerows(data)


class MapsMasstransitCurrentRoutesReportResource(sdk2.Resource):
    end_time = sdk2.parameters.String("End time")


class MapsMasstransitCurrentRoutesReport(sdk2.Task):
    class Parameters(sdk2.Parameters):
        window_to_create_report = sdk2.parameters.Integer(
            "Window size to create report for in mins",
            default=15
        )

    def on_execute(self):
        from yql.api.v1.client import YqlClient
        from library.python import resource as resource_manager
        import yt.wrapper as yt

        yt.config.set_proxy(YT_CLUSTER)
        yt.config["token"] = sdk2.Vault.data(YT_TOKEN_KEY)
        yql_client = YqlClient(db=YT_CLUSTER, token=sdk2.Vault.data(YQL_TOKEN_KEY))

        number_of_tables = ceil_div(self.Parameters.window_to_create_report, LOGS_SPAN_MINUTES)
        table_names = sorted(yt.list(LOGS_PATH))[-number_of_tables:]

        yql_script_template = resource_manager.find(SCRIPT_TEMPLATE_KEY)
        yql_script = yql_script_template.format(
            table_names=table_names
        )
        query = yql_client.query(yql_script, syntax_version=1)

        logging.info("Running query")
        query.run()
        self.set_info("Waiting for <a href=\"{}\">query</a> to complete".format(
            query.share_url
        ), do_escape=False)
        query.get_results()

        if not query.is_success:
            logging.error("YQL request status: %s", query.status)
            if query.errors:
                logging.error(
                    "Request errors: %s",
                    "; ".join(map(str, query.errors))
                )
            raise errors.SandboxTaskFailureError("Query was unsuccessful")

        start_time = table_names[0]
        end_time = table_names[-1]
        resource = MapsMasstransitCurrentRoutesReportResource(
            self,
            description="{} - {}".format(start_time, end_time),
            path="report"
        )
        resource.end_time = end_time

        resource_data = sdk2.ResourceData(resource)
        path = str(resource_data.path)

        clids = set()
        for table in query:
            assert table.column_names[0] == "clid"
            fieldnames = table.column_names[1:]

            filename = None
            if "vehicle_type" in fieldnames:
                filename = "routes"
            elif "uuid" in fieldnames:
                filename = "uuids"
            else:
                raise RuntimeError(
                    "Unexpected resulting column names: {}" .format(
                        table.column_names
                    )
                )

            results_path = os.path.join(path, filename)
            paths.make_folder(results_path)

            for clid, iterator in groupby(table.get_iterator(), key=lambda l: l[0]):
                clids.add(clid)
                filename = os.path.join(results_path, "{}.csv".format(clid))
                _write_csv(
                    data=imap(lambda l: l[1:], iterator),
                    fieldnames=fieldnames,
                    path=filename
                )

        _write_report(resource_manager.find(INDEX_HTML_TEMPLATE_KEY), path, sorted(list(clids)))

        resource_data.ready()

        self.set_info(
            (
                "See <a href=\"https://proxy.sandbox.yandex-team.ru/{}/index.html\">"
                "report</a> to compare results with a previous state"
            ).format(resource.id),
            do_escape=False
        )
