import logging
import requests
import re
import json
from datetime import datetime, timedelta
import dateutil.parser
from collections import defaultdict, namedtuple
from sandbox import sdk2
from sandbox.projects.garden.common import config as garden_config
from sandbox.sandboxsdk.environments import PipEnvironment


DEFAULT_GARDEN_MANAGER = garden_config.server_hostname("stable")
GARDEN_MODULE_STATISTICS_URL = "http://{0}/garden/module_statistics/?module={1}&from={2}"
GARDEN_BUILD_HIERARCHY_URL = "http://{0}/garden/build_hierarchy/?module={1}&build_id={2}"
RELEASE_STAGES_URL = "http://release-log.nika.maps.dev.yandex.net/release-stages/{0}"

SRC_MODULE = "mtr_mpro_export_src"
KNOWN_MODULES = set([
    "mtr_mpro_export_src",
    "mtr_mpro_export",
    "masstransit",
    "masstransit_deployment",
])

DEFAULT_REPORT = "Maps_Plus_Beta/Masstransit/development/Masstransit_data_deployment2"


def parse_mpro_datetime(mpro_datetime):
    # Python 2.7 strptime does not support %z
    # so we assume UTC+3 timezone here
    FORMAT = "%Y-%m-%dT%H:%M:%S+0300"
    return datetime.strptime(mpro_datetime, FORMAT) - timedelta(hours=3)


def parse_branch(shipping_date):
    """Returns mpro data release branch
    Mpro branch is used for data versioning
    """
    match = re.match("\d{8}_\d+_(\d+)_\d+", shipping_date)
    # Mannually added builds can have malformed shipping_date.
    # Here we process all builds that ever were added including removed now.
    # So we need to skip builds with malformed shipping_date
    return match.group(1) if match is not None else None


def request_json(url):
    r = requests.get(url)
    r.raise_for_status()
    return r.json()


def _get_descendants(garden_manager, build):
    # target build is in response
    return request_json(GARDEN_BUILD_HIERARCHY_URL.format(garden_manager, build["name"], build["id"]))


def request_builds(garden_manager, from_time):
    logging.info("request_builds")

    data = request_json(GARDEN_MODULE_STATISTICS_URL.format(garden_manager, SRC_MODULE, from_time))

    result = []
    for build in data:
        complete_status = None
        if build.get("completed_at"):
            result.append(build)

    return result


def request_creation_date(mpro_branch):
    data = request_json(RELEASE_STAGES_URL.format(mpro_branch))
    for stage in data["stages"]:
        if stage["name"] == "create_stable":
            return parse_mpro_datetime(stage["start"])
    return None


# Currently we could have several build with same key
#  * same branch could be used in several shipping_dates
#  * same shipping_date could be added to several src builds
# So we are looking for the first build of every stage here
class Statistics(object):
    Times = namedtuple("Times", ["start_time", "finish_time"])

    def __init__(self):
        self.times = defaultdict(dict)

    def update_stage(self, src_key, module, times):
        self.times[src_key][module] = min(
            self.times[src_key].get(module, times),
            times)

    def add_builds(self, src_key, builds):
        for build in builds:
            module = build["name"]

            if module not in KNOWN_MODULES:
                continue

            completed_at = build.get("completed_at")
            if not completed_at:
                continue

            times = Statistics.Times(
                dateutil.parser.parse(build["started_at"]),
                dateutil.parser.parse(completed_at)
            )

            self.update_stage(src_key, module, times)


def build_statistics(garden_manager, builds):
    logging.info("build_statistics")
    stats = Statistics()
    for build in builds:
        mpro_branch = parse_branch(build["properties"]["shipping_date"])
        if mpro_branch is not None:
            build_descedants = _get_descendants(garden_manager, build)
            stats.add_builds(mpro_branch, build_descedants)
    return stats.times


def make_fielddate(source_date):
    return datetime(
        source_date.year,
        source_date.month,
        source_date.day,
        source_date.hour,
        source_date.minute,
    ).isoformat()


def make_report(shippings_stats, last_days=30):
    logging.info("make_report")
    now = datetime.utcnow()
    values = []
    for mpro_branch, stages in shippings_stats.iteritems():
        # Skip branches older than 30 days
        # to reduce number of requests to mpro api
        min_start_time = min(t.start_time for t in stages.itervalues())
        if (min_start_time < now - timedelta(days=last_days)):
            continue
        source_date = request_creation_date(mpro_branch)
        if not source_date:     # old branches don't have creation dates
            continue
        fielddate = make_fielddate(source_date)
        for stage, times in stages.iteritems():
            values.append({
                "fielddate": fielddate,
                "name": SRC_MODULE,
                "branch": mpro_branch,
                "stage": stage,
                "finish_time": int((times.finish_time - source_date).total_seconds()),
                "start_time": int((times.start_time - source_date).total_seconds())
            })
    return {"values": values}


class MapsMasstransitDataDeploymentStatistics(sdk2.Task):
    """Calculates Masstransit data deployment statistics
    and uploads report to statface.
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment("python-statface-client", use_wheel=True)
        ]

    class Parameters(sdk2.Task.Parameters):
        description = "Update Masstransit data deployment statistics"
        last_days = sdk2.parameters.Integer(
            "Last days", default=30, required=True)
        garden_manager = sdk2.parameters.String(
            "Garden manager", default=DEFAULT_GARDEN_MANAGER, required=True)
        statface_report = sdk2.parameters.String(
            "Statface report", default=DEFAULT_REPORT)
        statface_beta = sdk2.parameters.Bool(
            "Statface beta", default=True)

    def _post_report(self, data):
        # statface_client is not available outside of the task environment
        import statface_client as sc
        host = (sc.STATFACE_BETA if self.Parameters.statface_beta
            else sc.STATFACE_PRODUCTION)
        client = sc.StatfaceClient(client_config={
            "host": host,
            "oauth_token": sdk2.Vault.data("robot-mtr_statface_token")
        })
        report = client.get_report(self.Parameters.statface_report)
        report.upload_data(scale=sc.MINUTELY_SCALE, data=json.dumps(data))

    def on_execute(self):
        from_time = (datetime.now() - timedelta(days=self.Parameters.last_days)).isoformat()
        builds = request_builds(self.Parameters.garden_manager, from_time)
        shippings_stats = build_statistics(self.Parameters.garden_manager, builds)
        report = make_report(shippings_stats, self.Parameters.last_days)

        if self.Parameters.statface_report:
            self._post_report(report)
