# -*- coding: utf-8 -*-

import logging
import requests
import StringIO
import tarfile

import sandbox.common.types.resource as ctr
from sandbox.projects.masstransit.common.utils import get_latest_resource
import sandbox.sdk2 as sdk2


MGT_FILES = [
    "CALENDAR",
    "ROUTES",
    "STOPS",
    "STOP_TIMES",
    "TRIPS",
    "TRIP_SHAPES",
    "TRIPS_STOPS",
    "SPECIAL_DAYS",
]

MGT_FILE_URL = "http://{host}/gtfs/file?name={file_name}&type=TEXT"
MGT_FILE_CHECKSUM_URL = "http://{host}/gtfs/file?name={file_name}&type=MD5"
REQUEST_TIMEOUT = 60  # Seconds.


def fetch_file(url):
    response = requests.get(url, timeout=REQUEST_TIMEOUT)
    response.raise_for_status()
    return response.content


def get_schedule_checksum(mgt_host):
    return "".join(
        fetch_file(MGT_FILE_CHECKSUM_URL.format(host=mgt_host, file_name=mgt_file))
        for mgt_file in MGT_FILES
    )


def add_file_to_tar(archive, name, content):
    info = tarfile.TarInfo(name)
    info.size = len(content)
    info.type = tarfile.REGTYPE
    archive.addfile(info, StringIO.StringIO(content))


def fetch_schedule(mgt_host, output_file_path):
    with tarfile.open(output_file_path, "w:gz") as archive:
        for mgt_file in MGT_FILES:
            add_file_to_tar(
                archive,
                "GTFS_{}".format(mgt_file),
                fetch_file(MGT_FILE_URL.format(host=mgt_host, file_name=mgt_file))
            )


class MapsMasstransitMgtScheduleResource(sdk2.Resource):
    """Archive with MGT schedule."""

    checksum = sdk2.parameters.String("Schedule checksum", required=True)


class MapsMasstransitFetchMGT(sdk2.Task):
    """Fetch MGT schedules."""

    class Parameters(sdk2.Parameters):
        kill_timeout = 10 * 60
        mgt_host = sdk2.parameters.String(
            "Host to fetch schedule from",
            required=True
        )
        resource_ttl = sdk2.parameters.Integer(
            "TTL of the output resource with schedule",
            required=True,
            default=20
        )

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        current_schedule_checksum = get_schedule_checksum(self.Parameters.mgt_host)
        last_saved_schedule = get_latest_resource(
            MapsMasstransitMgtScheduleResource,
            owner=self.owner,
            state=ctr.State.READY
        )

        if last_saved_schedule is None:
            logging.info("No schedules in sandbox, so fetching a new one.")
        elif current_schedule_checksum == last_saved_schedule.checksum:
            logging.info("Schedule checksum hasn't changed since the last time.")
            return
        else:
            logging.info("Schedule checksum has changed since the last time.")
            logging.info("Current schedule checksum: %s", current_schedule_checksum)
            logging.info("Last schedule checksum: %s", last_saved_schedule.checksum)

        resource_data = sdk2.ResourceData(MapsMasstransitMgtScheduleResource(
            self,
            "MGT schedule",
            "schedule.tar.gz",
            ttl=self.Parameters.resource_ttl,
            checksum=current_schedule_checksum
        ))
        fetch_schedule(self.Parameters.mgt_host, str(resource_data.path))
        resource_data.ready()
