# -*- coding: utf-8 -*-

import sandbox.sdk2 as sdk2
from datetime import datetime, timedelta
import os
import logging


DEFAULT_CLUSTER = 'hahn'
DEFAULT_SRC_PATH = '//logs/analyzer-dispatcher-signals-log/1d'
DEFAULT_DST_PATH = '//tmp/masstransit/filtered-signals'
DEFAULT_DAYS_BACK_TO_PROCESS = 67
DEFAULT_DAYS_TO_STORE = 70

DATETIME_FORMAT = '%Y-%m-%d'


def format_date(date):
    return date.strftime(DATETIME_FORMAT)


class MapsMasstransitFilterSignals(sdk2.Task):
    """
        Task to filter signals from traffic jams analyzer leaving only masstansit signals.
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 24 * 60 * 60

        yt_cluster = sdk2.parameters.String(
            'YT cluster',
            default=DEFAULT_CLUSTER
        )
        src_ypath = sdk2.parameters.String(
            'Path to the YT directory with analyzer logs',
            default=DEFAULT_SRC_PATH
        )
        dst_ypath = sdk2.parameters.String(
            'Path to the YT directory where filtered signals will be stored',
            default=DEFAULT_DST_PATH
        )
        days_to_process = sdk2.parameters.Integer(
            'Days back to process',
            default=DEFAULT_DAYS_BACK_TO_PROCESS
        )
        result_ttl_days = sdk2.parameters.Integer(
            'Result TTL',
            description='Sets TTL for resulting tables to store '
                        'them only for specified number of days',
            default=DEFAULT_DAYS_TO_STORE
        )
        force = sdk2.parameters.Bool(
            'Force filtering of existing tables',
            default=False
        )

    def on_execute(self):
        import yt.wrapper
        from maps.masstransit.tools.filter_signals.lib import filtration

        logging.getLogger('Yt').propagate = True

        yt.wrapper.config['token'] = sdk2.Vault.data('robot_robot-mtr_yt_token')
        yt.wrapper.config.set_proxy(self.Parameters.yt_cluster)
        yt.wrapper.config["auto_merge_output"]["action"] = "merge"
        yt.wrapper.mkdir(self.Parameters.dst_ypath, recursive=True)

        self.set_info('Starting table filtration.')

        dt = datetime.now() - timedelta(days=self.Parameters.days_to_process)
        end_date = datetime.now() - timedelta(days=1)
        while dt <= end_date:
            filtration.filter_signals(
                analyzer_signals_table=os.path.join(
                    self.Parameters.src_ypath,
                    format_date(dt)
                ),
                filtered_signals_table=os.path.join(
                    self.Parameters.dst_ypath,
                    format_date(dt)
                ),
                result_expiration_time=dt + timedelta(
                    days=self.Parameters.result_ttl_days
                ),
                force_filtering=self.Parameters.force
            )
            dt += timedelta(days=1)

        self.set_info('Finished.')
