import os
import logging
import datetime
from sandbox import sdk2
from sandbox import common
from collections import namedtuple
import sandbox.common.errors as errors
from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin


REQUEST = """
    $parse = DateTime::Parse("%Y-%m-%d");
    $days = 21;
    $stepSize = 300;
    $timeToFiveminFromMonday = ($time) -> {
        RETURN DateTime::ToMinutes(
            DateTime::FromSeconds(CAST($time as Uint32)) -
            DateTime::MakeTimestamp(DateTime::StartOfWeek(DateTime::FromSeconds(CAST($time AS Uint32))))
        ) / 5
    };
    $sortedTablesDesc = (
        SELECT
            ListTake(ListSortDesc(AGGREGATE_LIST(Path)), $days)
        FROM
            FOLDER("home/yatransport-prod/production/signals")
        WHERE
            Type = "table"
    );
    $endTableDate = DateTime::MakeDatetime($parse(ListLast(String::SplitToList(ListHead($sortedTablesDesc), '/'))));
    $endDate = $endTableDate + DateTime::IntervalFromDays(1);
    $startDate = $endTableDate - DateTime::IntervalFromDays($days);


    -- filter timestamp, because there are singals from the future
    INSERT INTO @filteredData
    WITH TRUNCATE
    SELECT
        DISTINCT clid, time, `uuid`
    FROM
        EACH($sortedTablesDesc)
    WHERE
        DateTime::FromSeconds(cast(time AS Uint32)) > DateTime::MakeTimestamp($startDate) AND
        DateTime::FromSeconds(cast(time AS Uint32)) < DateTime::MakeTimestamp($endDate)
    ;
    COMMIT;


    -- count signals in fivemin
    INSERT INTO @fiveMin
    WITH TRUNCATE
    SELECT
        clid,
        COUNT(*) AS signals,
        time
    FROM
        @filteredData
    GROUP BY
        clid, (time / $stepSize) * $stepSize AS time
    ;
    COMMIT;


    -- calculate signals in 15 minutes window
    -- to do that we have to take 3 previous fivemins without current one
    INSERT INTO @windowSum
    WITH TRUNCATE
    SELECT
        clid,
        SUM(signals) OVER w AS signals,
        time
    FROM
        @fiveMin
    WINDOW w as(
        PARTITION BY clid
        ORDER BY time
        ROWS BETWEEN 3 PRECEDING AND 1 PRECEDING
    );
    COMMIT;


    -- calc median
    INSERT INTO @medians
    WITH TRUNCATE
    SELECT
        clid,
        time_key,
        CASE
            WHEN ListLength(signals) > 2 THEN ListAggregate(signals, AGGREGATION_FACTORY("MEDIAN"))
            ELSE ListMin(signals)
        END AS `median`
    FROM
        (
        SELECT
            clid,
            time_key,
            AGGREGATE_LIST(signals) AS signals
        FROM
            @windowSum
        GROUP BY
            clid, $timeToFiveminFromMonday(time) AS time_key
        )
    WHERE ListLength(signals) > 1
    ;


    -- calc maximum by clid
    INSERT INTO @maxByClid
    WITH TRUNCATE
    SELECT
        clid,
        MAX(signals) as `max`
    FROM
        @windowSum
    GROUP BY
        clid
    ;
    COMMIT;


    INSERT INTO @statistics
    WITH TRUNCATE
    SELECT
        medians.time_key AS time_key,
        medians.clid AS clid,
        medians.`median` AS `median`,
        maximums.`max` AS `max`
    FROM
        @medians AS medians JOIN @maxByClid as maximums ON medians.clid == maximums.clid
    ;
    COMMIT;


    SELECT
        time_key,
        clid,
        CASE
            WHEN `median` > 0.5 * `max` THEN 0.8 * `median`
            WHEN `median` > 30 THEN 0.2 * `median`
            ELSE 1
        END AS expected_signals
    FROM
        @statistics
    ;
"""

YT_CLUSTER = "hahn"
TOKEN_KEY = "robot_robot-mtr_yql_token"
DATASET = "yandex-maps-masstransit-signal-statistics"
EXPECTED_VALUES_FILENAME = "expected_values.fbs"
CLIDS_CONSIDER_HOLIDAY_IN = ["moscow_new", "saint_petersburg", "1651901"]
CALENDAR_DATASET = "yandex-maps-calendar"
CALENDAR_DIR_PATH = "yandex-maps-calendar"
LIGHT_CALENDAR_NAME = "light_calendar.fb"
HOLIDAY_DECREASE_INDEX = 0.6
RUSSIA_REGION_ID = 225

Config = namedtuple("Config", [
    "ecstatic_env", "tvm_id", "tvm_secret_id"
])

ENVIRONMENT_CONFIGS = {
    "unstable": Config(
        ecstatic_env="unstable",
        tvm_id="2020679",  # maps-core-masstransit
        tvm_secret_id="sec-01e9za2yfnh2e2rzs8f4yw33zz"
    ),
    "datatesting": Config(
        ecstatic_env="datatesting",
        tvm_id="2020679",  # maps-core-masstransit
        tvm_secret_id="sec-01e9za2yfnh2e2rzs8f4yw33zz"
    ),
    "production": Config(
        ecstatic_env="stable",
        tvm_id="2020679",  # maps-core-masstransit
        tvm_secret_id="sec-01e9za2yfnh2e2rzs8f4yw33zz"
    )
}


class MapsMasstransitHistoricalRealtimeSignals(MapsEcstaticToolMixin, sdk2.Task):

    class Parameters(sdk2.Parameters):

        environment = sdk2.parameters.String(
            "Environment",
            choices=[(k, k) for k in ENVIRONMENT_CONFIGS.iterkeys()],
            default="unstable")

    def _time_key_to_date(self, now, time_key):
        monday = now - datetime.timedelta(days=now.weekday())
        return monday + datetime.timedelta(minutes=time_key * 5)

    def _get_last_calendar_dataset(self, config):
        datasets = self.ecstatic(
            "stable",
            ["versions", CALENDAR_DATASET],
            tvm_id=config.tvm_id,
            tvm_secret_id=config.tvm_secret_id
        ).splitlines()
        for dataset in reversed(datasets):
            if dataset:
                return dataset
        raise RuntimeError("Can't find calendar dataset")

    def _download_from_ecstatic(self, dataset, path, config):
        self.ecstatic(
            "stable",
            ["download", dataset, "--out", path],
            tvm_id=config.tvm_id,
            tvm_secret_id=config.tvm_secret_id
        )

    def _fetch_calendar(self):
        config = ENVIRONMENT_CONFIGS[self.Parameters.environment]
        dataset = self._get_last_calendar_dataset(config)
        self._download_from_ecstatic(dataset, CALENDAR_DIR_PATH, config)

    def _execute_query(self, yql_client):
        query = yql_client.query(REQUEST, syntax_version=1)
        query.run()
        query.get_results()

        if not query.is_success:
            logging.error("YQL request status: {0}".format(query.status))
            if query.errors:
                logging.error("Request errors: {0}".format(
                    "; ".join(map(str, query.errors))))
            raise errors.TaskFailure("YQL query failed")

        return query

    def _is_working_day(self, date):
        return (date.weekday() not in [5, 6])

    def _save_dataset(self, query_result, data_dir):
        from maps.masstransit.libs.realtime_signals_monitoring.py.expected_values import PyExpectedValues
        from maps.analyzer.libs.light_calendar.py.calendar import PyCalendarStorage

        self._fetch_calendar()

        expected_values = PyExpectedValues()
        calendar = PyCalendarStorage(filename=os.path.join(CALENDAR_DIR_PATH, LIGHT_CALENDAR_NAME), populate=True, lockMemory=True)
        now = datetime.date.today()
        for time_key, clid, signals in query_result.table.get_iterator():
            if clid in CLIDS_CONSIDER_HOLIDAY_IN:
                date = self._time_key_to_date(now, time_key)
                if calendar.is_holiday(date, RUSSIA_REGION_ID) and self._is_working_day(date):
                    signals *= HOLIDAY_DECREASE_INDEX
            expected_values.add_item(time_key, clid, signals)

        expected_values.save(str(data_dir / EXPECTED_VALUES_FILENAME))

    def _upload_to_ecstatic(self, data_dir):
        config = ENVIRONMENT_CONFIGS[self.Parameters.environment]
        version = datetime.datetime.now().strftime("%Y-%m-%d")
        self.ecstatic(
            config.ecstatic_env,
            ["upload", DATASET + "=" + version, str(data_dir), "+stable"],
            tvm_id=config.tvm_id,
            tvm_secret_id=config.tvm_secret_id
        )

    def on_execute(self):
        from yql.api.v1.client import YqlClient

        yql_client = YqlClient(db=YT_CLUSTER, token=sdk2.Vault.data(TOKEN_KEY))
        result = self._execute_query(yql_client)
        data_dir = self.path("expected_values_dataset")
        common.fs.make_folder(data_dir)

        self._save_dataset(result, data_dir)
        self._upload_to_ecstatic(data_dir)
