# -*- coding: utf-8 -*-

import os
import StringIO
import tarfile
import requests
import sandbox
import gkuop
import shutil
import logging
import sandbox.sdk2 as sdk2
from sandbox.sandboxsdk import errors
import sandbox.common.types.resource as ctr
from sandbox.projects.masstransit.common.utils import get_latest_resource


CLID = "moscow"
CODD_FILES = ["TSmgt", "TScomp"]
MGT_FILES = ["ROUTES", "VEHICLE_TASK"]
GKUOP_FILENAME = "gkuop.csv"
ENVIRONMENTS = ["unstable", "testing", "stable"]


class MapsMasstransitVehicleTasksResource(sdk2.Resource):
    environment = sdk2.parameters.String("Environment")


class MapsMasstransitVehicleTasksSeparatedResource(sdk2.Resource):
    environment = sdk2.parameters.String("Environment")
    filename = sdk2.parameters.String("File name")


def write_file(data, filename):
    with open(filename, "w") as f:
        f.write(data)


def fetch_file(url):
    from retry.api import retry_call

    def try_fetch():
        response = requests.get(url)
        response.raise_for_status()
        return response

    response = retry_call(try_fetch, tries=10, delay=10)
    return response.content


def check_if_file_is_empty(filename):
    if os.path.getsize(filename) == 0:
        raise Exception("File {} is empty".format(filename))


class MapsMasstransitImportVehicleTasks(sdk2.Task):
    """
    A single vehicle can be assigned to different routes during the day.
    These scheduled assignments are called vehicle tasks.
    This task imports vehicle tasks from a provider, parses it and stores result
    in a resource.
    """

    def _try_write_latest_resource(self, filename, filepath):
        latest_resource = get_latest_resource(
            MapsMasstransitVehicleTasksSeparatedResource,
            attrs=dict(filename=filename, environment=self.Parameters.environment),
            state=ctr.State.READY
        )
        if latest_resource is None:
            raise Exception("Can't find latest resource for {}".format(filename))
        latest_resource = sdk2.ResourceData(latest_resource)
        shutil.copy(str(latest_resource.path), str(filepath))

    def _create_resource(self, filename, filepath):
        resource = sdk2.ResourceData(MapsMasstransitVehicleTasksSeparatedResource(
            self,
            "{} resource".format(filename),
            filename,
            environment=self.Parameters.environment,
            filename=filename
        ))
        shutil.copy(str(filepath), str(resource.path))
        resource.ready()

    class Parameters(sdk2.Parameters):
        kill_timeout = 10 * 60

        gkuop_url = sdk2.parameters.String(
            "Url of FTP server with vehicle tasks",
            default="gftp.gkuop.ru"
        )

        codd_info_url = sdk2.parameters.String(
            "Url template of CODD info source",
            default="http://core-masstransit-partners-proxy.maps.yandex.net/{}"
        )

        mgt_host = sdk2.parameters.String(
            "MGT data host",
            default="mgt-proxy2.sas.yp-c.yandex.net"
        )

        environment = sdk2.parameters.String(
            "Environment",
            choices=[(k, k) for k in ENVIRONMENTS],
            default="unstable")

    class Requirements(sdk2.Requirements):
        cores = 1
        dns = sandbox.common.types.misc.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

    @staticmethod
    def write_gkuop_csv(rows, filename):
        import unicodecsv as csv

        with open(filename, "w") as f:
            writer = csv.writer(
                f, delimiter=";",
                encoding="utf-8"
            )
            writer.writerow(gkuop.FIELD_NAMES)  # header
            writer.writerows(rows)

    @staticmethod
    def _convert_csv(src, dst):
        import unicodecsv as csv

        """
        MGT doesn't exactly follow CSV format as specified by http://www.ietf.org/rfc/rfc4180.txt
        This function converts MGT's field quoting style to the correct one.
        """
        reader = csv.reader(src, delimiter=";", quoting=csv.QUOTE_NONE)
        writer = csv.writer(dst, delimiter=";", quoting=csv.QUOTE_MINIMAL)
        writer.writerows(reader)

    def _write_gkuop(self):
        from retry.api import retry_call
        gkuop_path = os.path.join(CLID, GKUOP_FILENAME)
        credentials = sdk2.yav.Secret(
            "sec-01e5aj64wjzqw0twd1har6yzfq",
            "ver-01e5aj64x1g02spvwdg5qz2g9w"
        ).data()
        try:
            rows = retry_call(
                gkuop.fetch_data,
                fargs=[self.Parameters.gkuop_url, credentials],
                tries=10, delay=10
            )
            self.write_gkuop_csv(rows, gkuop_path)
            check_if_file_is_empty(gkuop_path)
            self._create_resource(GKUOP_FILENAME, gkuop_path)
        except Exception as e:
            logging.info("Exception during downloading gkuop file: %s", e)
            self.failed_to_download_files.append(GKUOP_FILENAME)
            self._try_write_latest_resource(GKUOP_FILENAME, gkuop_path)

    def on_execute(self):
        from sandbox.projects.masstransit.MapsMasstransitFetchMGT import MGT_FILE_URL

        self.failed_to_download_files = []

        subdir_name = CLID
        os.mkdir(subdir_name)

        self._write_gkuop()

        for filename in CODD_FILES:
            filepath = os.path.join(subdir_name, "{}.json".format(filename))
            try:
                contents = fetch_file(url=self.Parameters.codd_info_url.format(filename))
                write_file(contents, filepath)
                check_if_file_is_empty(filepath)
                self._create_resource("{}.json".format(filename), filepath)
            except Exception as e:
                logging.info("Exception while downloading %s file: %s", filename, e)
                self.failed_to_download_files.append(filename)
                self._try_write_latest_resource("{}.json".format(filename), filepath)

        for filename in MGT_FILES:
            filepath = os.path.join(subdir_name, filename)
            try:
                contents = fetch_file(url=MGT_FILE_URL.format(
                    host=self.Parameters.mgt_host,
                    file_name=filename
                ))
                with open(filepath, "w") as dst:
                    src = StringIO.StringIO(contents)
                    self._convert_csv(src, dst)
                check_if_file_is_empty(filepath)
                self._create_resource(filename, filepath)
            except Exception as e:
                logging.info("Exception while downloading %s file: %s", filename, e)
                self.failed_to_download_files.append(filename)
                self._try_write_latest_resource(filename, filepath)

        resource_data = sdk2.ResourceData(MapsMasstransitVehicleTasksResource(
            self,
            "Masstransit vehicle tasks",
            "vehicle_tasks.tar.gz",
            environment=self.Parameters.environment
        ))
        with tarfile.open(str(resource_data.path), "w:gz") as tar:
            tar.add(subdir_name)
        resource_data.ready()
        if self.failed_to_download_files:
            raise errors.SandboxTaskFailureError("Can't download files: {}".format(self.failed_to_download_files))
