# -*- coding: utf-8 -*-

from collections import namedtuple
from datetime import datetime
import re


FsNodeEntry = namedtuple("FsNodeEntry", [
    "name",
    "is_dir",
    "modification_time",
])


def parse_microsoft_dir_entry(s):
    """
    Unfortunately, some FTP servers don't support MLSD command. The only way to
    get information about directory contents from such servers is parsing
    human-readable output which varies depending on server platform.
    This function parses an output line of LIST command as returned by
    Microsoft FTP server.
    """

    # Sample input:
    # 02-21-20  11:19AM       <DIR>          FTP-ISRPD
    m = re.match("(\S+  \S+)\s+(\S+)\s+(.*)", s)
    if m is None:
        raise ValueError("Invalid dir entry format: {}".format(s))
    modification_time_str, type_, name = m.groups()
    modification_time = datetime.strptime(modification_time_str, "%m-%d-%y  %I:%M%p")
    return FsNodeEntry(name, type_ == "<DIR>", modification_time)


def parse_unix_dir_entry(s, year=datetime.now().year):
    """
    Unfortunately, some FTP servers don't support MLSD command. The only way to
    get information about directory contents from such servers is parsing
    human-readable output which varies depending on server platform.
    This function parses a unix-style output line of LIST command.
    """

    from dateutil.parser import parse as parse_datetime

    # Sample input:
    # -rwxrwxrwx   1 owner    group      1708252896 Apr  2  2019 aclfile
    m = re.match("(\S{10})\s+(?:\S+\s+){4}(\S+\s+\S+\s+\S+)\s+(.*)", s)
    if m is None:
        raise ValueError("Invalid dir entry format: {}".format(s))
    permissions, modification_time_str, name = m.groups()

    modification_time = parse_datetime(modification_time_str, default=datetime(year, 1, 1))

    return FsNodeEntry(name, permissions[0] == "d", modification_time)


def list_dir(ftp, path):
    entries = []
    ftp.dir(
        path,
        lambda dir_entry: entries.append(parse_unix_dir_entry(dir_entry))
    )
    return entries


def get_subdir_names(ftp, directory):
    return [entry.name for entry in list_dir(ftp, directory) if entry.is_dir]


def fetch_raw_file(ftp, filename):
    data = bytearray()
    ftp.retrbinary("RETR {}".format(filename), data.extend)
    return data
