# -*- coding: utf-8 -*-
"""
Code for parsing data from "ГКУ Организатор Перевозок"
"""

import os
import logging
import ftplib
import io
from datetime import datetime, timedelta

import ftp_helpers


def _find_latest_vehicle_tasks(ftp, root="/FTP-ISRPD", max_latest_files=4):
    filenames = []
    for subdir in ftp_helpers.get_subdir_names(ftp, root):
        try:
            search_dir = os.path.join(root, subdir, "SOURCE/ORDERS")
            items = ftp_helpers.list_dir(ftp, search_dir)
            latest_csvs = sorted(
                [item for item in items if item.name.endswith(".csv")],
                key=lambda item: item.modification_time,
                reverse=True
            )[:max_latest_files]
            for item in latest_csvs:
                filenames.append(os.path.join(search_dir, item.name))
        except ftplib.error_perm:
            logging.info("Could not find vehicle tasks in {}".format(subdir))
    return filenames


def _parse_time(s, date, time_format="%H:%M"):
    time = datetime.strptime(s[:len(time_format)], time_format).time()
    return datetime.combine(date, time)


def _convert_row(row):
    DATE_FORMAT = "%d.%m.%Y"
    OUTPUT_TIME_FORMAT = "%Y%m%d %H:%M"

    date = datetime.strptime(row[u"Сутки"], DATE_FORMAT)
    start_time = _parse_time(row[u"Начало"], date)
    end_time = _parse_time(row[u"Окончание"], date)
    if start_time >= end_time:
        end_time += timedelta(days=1)

    route = row[u"Выход_ид"].split("_")[0]
    vehicle_id = row[u"ГРЗ"].lower().replace(" ", "")

    return (
        route,
        vehicle_id,
        start_time.strftime(OUTPUT_TIME_FORMAT),
        end_time.strftime(OUTPUT_TIME_FORMAT),
    )


def _create_ftp_client(url, credentials):
    return ftplib.FTP(
        url,
        credentials["login"],
        credentials["password"],
        timeout=120
    )


def fetch_data(url, credentials):
    import unicodecsv as csv

    ftp = _create_ftp_client(url, credentials)
    filenames = _find_latest_vehicle_tasks(ftp, "/FTP-ISRPD")

    result_rows = set()
    for filename in filenames:
        logging.info("Parsing %s", filename)

        # FTP client is recreated because a connection tends to hang up after
        # numerous requests.
        ftp = _create_ftp_client(url, credentials)

        try:
            contents = ftp_helpers.fetch_raw_file(ftp, filename)
            buffer = io.BytesIO(contents)
            reader = csv.DictReader(buffer, delimiter=";", encoding="utf-8")

            for row in reader:
                result_rows.add(_convert_row(row))
        except Exception as e:
            logging.exception("Could not parse %s", filename)

    logging.info("Got %d entries", len(result_rows))

    return sorted(list(result_rows))


FIELD_NAMES = ["route", "vehicle_id", "start_time", "end_time"]
