# -*- coding: utf-8 -*-

from collections import defaultdict
import logging

import sandbox.sdk2 as sdk2
from default_config import DEFAULT_CONFIG


DEFAULT_REPORT = 'Maps/Others/Jumps'


class MapsMasstransitJumpCounter(sdk2.Task):
    """
        Task for measuring quality of mtinfo prognoses.
        See MTDEV-72 for details.
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 15 * 60

        statface_path = sdk2.parameters.String(
            'Path to statface report', default=DEFAULT_REPORT
        )
        statface_beta = sdk2.parameters.Bool("Use stat-beta", default=True)
        config = sdk2.parameters.JSON("Config", default=DEFAULT_CONFIG)

    def on_execute(self):
        logging.basicConfig(level=logging.DEBUG)

        import count_jumps
        from maps.masstransit.pylibs.statistics.lib import statface

        now = statface.current_time_string('minutely')
        results = []
        for region in self.Parameters.config['regions']:
            self.set_info('Measuring {}'.format(region['name']))
            jumps, measurments = map(self._sum_by_clid, count_jumps.measure(self.Parameters.config, region))
            region_count = 0
            region_total = 0
            for clid in measurments.iterkeys():
                region_count += jumps[clid]
                region_total += measurments[clid]
                results.append({
                    'fielddate': now,
                    'id': '\t{region}\t{clid}\t'.format(region=region['name'], clid=clid),
                    'jumps': jumps[clid],
                    'total': measurments[clid]
                })
            results.append({
                'fielddate': now,
                'id': '\t{region}\t'.format(region=region['name']),
                'jumps': region_count,
                'total': region_total
            })

        self.set_info('Uploading results')
        client = statface.statface_client(
            token=sdk2.Vault.data('robot-mtr_statface_token'),
            use_beta=self.Parameters.statface_beta
        )
        statface.post_report(
            client,
            results,
            'minutely',
            self.Parameters.statface_path
        )

        self.set_info('Finished')

    @staticmethod
    def _sum_by_clid(counts):
        result = defaultdict(int)
        for key, value in counts.iteritems():
            clid = key.split('|')[0]
            result[clid] += value
        return result
