from sandbox import sdk2
from sandbox.projects.masstransit.common.shooting import Target, make_load_config, make_step_rps
from sandbox.projects.masstransit.common.subtasks import ParentTask, SubtasksError
from sandbox.projects.masstransit.common.utils import determine_nanny_hosts
from sandbox.projects.masstransit.common.constants import (
    MTINFO_LOAD_NANNY_SERVICE_ID
)
from sandbox.projects.masstransit.MapsMasstransitPrepareProductionLogAmmo import MapsMasstransitPrepareProductionLogAmmo
from sandbox.projects.masstransit.MapsMasstransitRestartService import MapsMasstransitRestartService
from sandbox.projects.masstransit.resources import MapsMasstransitProductionLogAmmo
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from itertools import izip
import json

from config import (
    MTINFO_HEADERS,
    MTINFO_AUTOSTOP
)


class MapsMasstransitMtinfoShooting(ParentTask):

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        skip_mtinfo_light = sdk2.parameters.Bool('Skip mtinfo-light', default=False)
        skip_mtinfo_all = sdk2.parameters.Bool('Skip mtinfo-all', default=False)
        skip_mtinfo_heavy = sdk2.parameters.Bool('Skip mtinfo-heavy', default=False)
        restart_service = sdk2.parameters.Bool('Restart service', default=True)

    def _prepare_mtinfo_ammo(
        self, component,
        handles, inverse_filter=False, requests_number=1000000
    ):
        stage = '{0}_generate_mtinfo_ammo'.format(component)
        ammo_task = self.run_subtask(
            stage,
            lambda: MapsMasstransitPrepareProductionLogAmmo(
                self,
                description=stage,
                notifications=self.Parameters.notifications,
                resource_type='MAPS_MASSTRANSIT_PRODUCTION_LOG_AMMO',
                nanny_service='maps_core_masstransit_mtinfo_load',
                vhost='core-masstransit-mtinfo.maps.yandex.net',
                requests_number=requests_number,
                handles=handles,
                inverse_filter=inverse_filter
            )
        )
        return ammo_task.Parameters.ammo

    def _shoot(self, component, targets, is_const, autostop):
        config_content = make_load_config(
            task='MTDEV-305',
            component=component,
            targets=targets,
            is_const=is_const,
            autostop=autostop
        )

        return ShootViaTankapi(
            self,
            description='Shooting {0}'.format(component),
            notifications=self.Parameters.notifications,
            use_public_tanks=False,
            nanny_service=MTINFO_LOAD_NANNY_SERVICE_ID,
            tanks=['nanny:maps_core_tanks_load'],
            config_source='file',
            config_content=config_content,
            use_monitoring=True,
            monitoring_source='arcadia',
            monitoring_arc_path='maps/masstransit/info/mtinfo/regression/telegraf_config.cfg',
            ammo_source='in_config'
        )

    def _shoot_mtinfo(self, name, rps, ammo_resource, is_const=False):
        mtinfo_targets = determine_nanny_hosts(MTINFO_LOAD_NANNY_SERVICE_ID)
        return self._shoot(
            component='{0} (mtinfo)'.format(name),
            targets=[
                Target(target, rps, ammo_resource, MTINFO_HEADERS)
                for target in mtinfo_targets
            ],
            is_const=is_const,
            autostop=MTINFO_AUTOSTOP
        )

    def _do_run_load_test(
        self,
        name,
        mtinfo_rps,
        mtinfo_handles,
        inverse_filter=False
    ):
        mtinfo_ammo_resource = self._prepare_mtinfo_ammo(
            name,
            mtinfo_handles,
            inverse_filter
        )

        # Need nested function here to create new tasks
        # only if they were not created before
        def shoot_subtasks_factory():
            subtasks = []
            subtasks.append(self._shoot_mtinfo(
                name,
                mtinfo_rps,
                mtinfo_ammo_resource
            ))

            return subtasks

        self.run_subtasks(
            '{0}_shoot'.format(name),
            shoot_subtasks_factory
        )

    def _run_load_test(self, *args, **kwargs):
        try:
            self._do_run_load_test(*args, **kwargs)
        except SubtasksError:
            # Do not fail the parent task, continue with next load tests
            pass


    def _restart_mtinfo(self):
        self.run_subtask(
            'Restart MtInfo',
            lambda: MapsMasstransitRestartService(
                self,
                description='Restarting MtInfo before load tests',
                nanny_service_id='maps_core_masstransit_mtinfo_load'
            )
        )

    def on_execute(self):

        if self.Parameters.restart_service:
            self._restart_mtinfo()

        if not self.Parameters.skip_mtinfo_light:
            self._run_load_test(
                name='mtinfo-light',
                mtinfo_rps=[make_step_rps(12000)],
                mtinfo_handles=[
                    '/v2/line',
                    '/v2/thread',
                    '/mtinfo/stop',
                    '/v2/stop_internal',
                ]
            )

        # We cannot test /mtinfo/prognosis/vehicle here because
        # vehicle ids from production log will always cause 404
        mtinfo_all_handles=['^/(mtinfo/prognosis|v2)/vehicle*$']
        if not self.Parameters.skip_mtinfo_all:
            self._run_load_test(
                name='mtinfo-all',
                mtinfo_rps=[make_step_rps(12000)],
                mtinfo_handles=mtinfo_all_handles,
                inverse_filter=True
            )

        # There is no /mtinfo/prognosis/vehicle because of 404 here too
        mtinfo_heavy_handles=[
                '/mtinfo/prognosis/stop',
                '/v2/stop',
                '/v2/schedule',
                '/v2/trajectories',
            ]

        if not self.Parameters.skip_mtinfo_heavy:
            self._run_load_test(
                name='mtinfo-heavy',
                mtinfo_rps=[make_step_rps(15000)],
                mtinfo_handles=mtinfo_heavy_handles
            )
