# -*- coding: utf-8 -*-

import logging

import sandbox.common.errors
import sandbox.common.platform
import sandbox.common.types.resource
import sandbox.common.urls
import sandbox.sdk2


class MapsMasstransitMtmatrixS3Cleanup(sandbox.sdk2.Task):
    """Delete old objects in s3 objects"""

    class Requirements(sandbox.sdk2.Requirements):
        cores = 1
        disk_space = 1024  # MB
        ram = 150 * 1024  # MB

    class Parameters(sandbox.sdk2.Task.Parameters):
        kill_timeout = 15 * 60  # seconds

        with sandbox.sdk2.parameters.RadioGroup('S3 instance') as s3_instance:
            s3_instance.values["stable"] = s3_instance.Value("stable", default=True)
            s3_instance.values["testing"] = s3_instance.Value("testing")

        buckets = sandbox.sdk2.parameters.List(
            "S3 buckets to clean",
            default=['matrix-result-testing'],
            required=True)

        max_age_seconds = sandbox.sdk2.parameters.Integer(
            "Max age of an object (seconds)",
            default=5000,
            required=True)

        client_tvm_id = sandbox.sdk2.parameters.Integer("S3 client TVM id")
        tvm_secret = sandbox.sdk2.parameters.YavSecret("S3 client TVM secret")

    def _client_tvm_id(self):
        return self.Parameters.client_tvm_id

    def _tvm_secret(self):
        return self.Parameters.tvm_secret.data()["client_secret"]

    S3_SETTINGS = {
        'stable': {
            'endpoint': 'https://s3.mds.yandex.net',
            'tvm_id': 2017579
        },
        'testing': {
            'endpoint': 'https://s3.mdst.yandex.net',
            'tvm_id': 2017577
        }
    }

    def _s3_endpoint(self):
        return self.S3_SETTINGS[self.Parameters.s3_instance]['endpoint']

    def _s3_tvm_id(self):
        return self.S3_SETTINGS[self.Parameters.s3_instance]['tvm_id']

    def on_create(self):
        if self.Requirements.tasks_resource is None:
            # This code does not seem to work when running a task locally with
            # --enable-taskbox. The if-else solves this.
            logging.info("task binary is not set, looking for latest suitable one")
            self.Requirements.tasks_resource = (
                sandbox.sdk2.service_resources.SandboxTasksBinary.find(
                    attrs={"target": "masstransit/MapsMasstransitMtmatrixS3Cleanup"},
                    state=sandbox.common.types.resource.State.READY
                ).first()
            )
            logging.info("task binary: {}".format(self.Requirements.tasks_resource))
        else:
            logging.info("task binary is already set")

    def on_execute(self):
        logging.info("checking that tasks resource is properly set")
        logging.info("current tasks_resource: {}".format(
            self.Requirements.tasks_resource))
        if self.Requirements.tasks_resource is None:
            raise sandbox.common.errors.TaskFailure(
                "self.Requirements.tasks_resource is not set"
                " for MapsMasstransitMtmatrixS3Cleanup")

        from maps.masstransit.tools.s3cleanup.lib.s3cleanup import make_s3_client, s3_cleanup

        s3_client = make_s3_client(
            self_tvm_id=self._client_tvm_id(),
            self_tvm_secret=self._tvm_secret(),
            dst_tvm_id=self._s3_tvm_id(),
            s3_endpoint=self._s3_endpoint())

        for bucket in self.Parameters.buckets:
            logging.info('Cleaning bucket {}...'.format(bucket))
            s3_cleanup(
                s3_client=s3_client,
                bucket=bucket,
                max_age_seconds=self.Parameters.max_age_seconds)
