from sandbox import sdk2
from sandbox.projects.masstransit.common.shooting import make_simple_load_config, make_step_rps
from sandbox.projects.masstransit.common.subtasks import ParentTask, SubtasksError
from sandbox.projects.masstransit.common.utils import make_shooting_stage
from sandbox.projects.masstransit.MapsMasstransitPrepareProductionLogAmmo import MapsMasstransitPrepareProductionLogAmmo
from sandbox.projects.masstransit.MapsMasstransitRestartService import MapsMasstransitRestartService
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi

AMMO_RESOURCE_TYPE='MAPS_MASSTRANSIT_PRODUCTION_LOG_AMMO'
VHOST='core-masstransit-matrix.maps.yandex.net'


class MapsMasstransitMtmatrixShooting(ParentTask):

    class Parameters(sdk2.Parameters):
        restart_service = sdk2.parameters.Bool('Restart service', default=True)

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def generate_mtmatrix_ammo(self, handle, requests_number=1000000):
        stage = 'Ammo generator for {0} (mtmatrix)'.format(handle)
        ammo_task = self.run_subtask(
            stage,
            lambda: MapsMasstransitPrepareProductionLogAmmo(
                self,
                description=stage,
                notifications=self.Parameters.notifications,
                resource_type=AMMO_RESOURCE_TYPE,
                vhost=VHOST,
                requests_number=requests_number,
                handles=[handle]
            )
        )
        return ammo_task.Parameters.ammo

    def shoot_mtmatrix(self, handle, rps, ammo_resource, autostop, shooting_name_suffix, post_ammo=False, anonymous=False):
        headers = [
            'Host: core-masstransit-matrix.maps.yandex.net',
            'Accept: application/x-protobuf'
        ]

        if not anonymous:
            yav_secret = sdk2.yav.Secret('sec-01ercvecq6fhdmbss7qkb0nsgp')  # TVM secret for mtmatrix-testing
            headers += [
                'X-Ya-Service-Ticket: {0}'.format(yav_secret.data()["client_secret"])
            ]

        config_content = make_simple_load_config(
            task='MTDEV-836',
            component=handle,
            rps=rps,
            headers=headers,
            autostop=autostop,
            shooting_name_suffix=shooting_name_suffix,
            ammo_type='uripost' if post_ammo else 'uri'
        )

        stage = make_shooting_stage(handle, rps, shooting_name_suffix)

        self.run_subtask(
            stage,
            lambda: ShootViaTankapi(
                self,
                description='Shooting {0} with rps {1}'.format(handle, rps),
                notifications=self.Parameters.notifications,
                use_public_tanks=False,
                tanks=['nanny:maps_core_tanks_load'],
                nanny_service='maps_core_masstransit_matrix_load',
                config_source='file',
                config_content=config_content,
                use_monitoring=True,
                monitoring_source='arcadia',
                monitoring_arc_path='maps/masstransit/matrix/regression/telegraf_config.cfg',
                ammo_source='resource',
                ammo_resource=ammo_resource
            )
        )

    def run_load_test(self, handle, mtmatrix_rps, ammo_resource=None, autostop=[],
                      shooting_name_suffix=None, post_ammo=False):
        try:
            if ammo_resource:
                mtmatrix_ammo_resource = ammo_resource
            else:
                mtmatrix_ammo_resource = self.generate_mtmatrix_ammo(handle)

            for rps in mtmatrix_rps:
                try:
                    self.shoot_mtmatrix(
                        handle,
                        rps,
                        mtmatrix_ammo_resource,
                        autostop,
                        shooting_name_suffix,
                        post_ammo,
                        anonymous=True
                    )
                except SubtasksError:
                    pass
        except SubtasksError:
            # Do not fail the parent task, continue with next load tests
            pass

    def restart_mtmatrix(self):
        self.run_subtask(
            'Restart MtMatrix',
            lambda: MapsMasstransitRestartService(
                self,
                description='Restarting MtMatrix before load tests',
                nanny_service_id='maps_core_masstransit_matrix_load'
            )
        )

    def on_execute(self):
        if self.Parameters.restart_service:
            self.restart_mtmatrix()

        self.run_load_test(
            handle='/masstransit/v2/matrix',
            mtmatrix_rps=[
                make_step_rps(max_rps=250)
            ]
        )

        self.run_load_test(
            handle='/masstransit/v2/matrix',
            mtmatrix_rps=[
                make_step_rps(max_rps=24, rps_step_size=1)
            ],
            autostop=[
                'quantile(50,800,20)',
                'http(5xx,10%,3)',
                'http(4xx,25%,4)',
                'net(101,25,5)'
            ],
            ammo_resource='1866198106',
            shooting_name_suffix='random matrices 10x10'
        )

        self.run_load_test(
            handle='/pedestrian/v2/matrix',
            mtmatrix_rps=[
                make_step_rps(max_rps=6000)
            ],
            shooting_name_suffix='(mtmatrix)'
        )

        self.run_load_test(
            handle='/pedestrian/v2/matrix',
            mtmatrix_rps=[
                make_step_rps(max_rps=320, min_rps=120, rps_step_size=20)
            ],
            ammo_resource='2010529460',
            shooting_name_suffix='random matrices 10x10'
        )
