# -*- coding: utf-8 -*-

import binascii
import logging
import os
import requests
import subprocess

import sandbox.common.errors
import sandbox.common.platform
import sandbox.common.types.resource
import sandbox.common.urls
import sandbox.sdk2

_JUGGLER_HOST_NAME_TEMPLATE = "sandbox.MapsMasstransitMtmatrixTester.{environment}"
_JUGGLER_SERVICE_NAME = "shooting_failed"

_TVMTOOL_PORT = 12346
_TVMTOOL_CACHE_DIR = "tvmtool-cache-dir"

_TVM_CONFIG = """{
    "clients": {
        "maps-core-masstransit-testing-tools": {
            "secret": "env:TVM_SECRET",
            "self_tvm_id": 2028046,
            "dsts": {
                "maps-core-masstransit-matrix": {
                    "dst_id": 2025067
                },
                "s3-mds": {
                    "dst_id": 2017579
                }
            }
        }
    }
}
"""


def _random_auth_token():
    """Generate a random string to be used as a TVM authorization token.

    More on this token:
    https://wiki.yandex-team.ru/passport/tvm2/tvm-daemon/#kakzapustitprocess
    """
    return binascii.hexlify(os.urandom(16)).decode("utf-8")


def _prepare_tvmtool():
    platform = sandbox.common.platform.platform()
    arch = sandbox.common.platform.get_arch_from_platform(platform)
    tvmtool_resource = sandbox.sdk2.Resource.find(
        type="TVM_TOOL_BINARY", arch=arch).first()
    return str(sandbox.sdk2.ResourceData(tvmtool_resource).path)


class MapsMasstransitMtmatrixTester(sandbox.sdk2.Task):
    """Create load on matrix router in testing environment."""

    # TODO(immartynov): Use one of the libraries:
    #   * sandbox/projects/common/juggler or
    #   * sandbox/projects/maps/common/juggler_alerts.py
    def _send_status_to_juggler(self, status, current_task_state):
        juggler_host_name = _JUGGLER_HOST_NAME_TEMPLATE.format(
            environment=self.Parameters.environment)
        task_url = sandbox.common.urls.get_task_link(self.id)
        event = {
            "host": juggler_host_name,
            "service": _JUGGLER_SERVICE_NAME,
            "status": status,
            "description": (
                "MapsMasstransitMtmatrixTester is in state {}: {}".format(
                    current_task_state, task_url)
            )
        }

        try:
            reply = requests.post(
                "http://juggler-push.search.yandex.net/events",
                json={"source": "sandbox", "events": [event]}
            )
            event_status = reply.json()["events"][0]
            if event_status["code"] != 200:
                raise Exception(event_status["error"])
        except Exception as e:
            logging.exception("could not send event to Juggler: {}".format(e))

    class Requirements(sandbox.sdk2.Requirements):
        cores = 1
        disk_space = 1024  # MB
        ram = 150 * 1024  # MB

    class Parameters(sandbox.sdk2.Task.Parameters):
        kill_timeout = 15 * 60  # seconds

        yav_secret_id = sandbox.sdk2.parameters.String("TVM secret ID in yav")

        with sandbox.sdk2.parameters.RadioGroup("Environment") as environment:
            environment.values.development = environment.Value(
                value="development", default=True)
            environment.values.stable = environment.Value(value="stable")

        thread_count = sandbox.sdk2.parameters.Integer("Number of threads to shoot", default=30)
        sync_request_count = sandbox.sdk2.parameters.Integer("Number of sync requests", default=1000)
        async_request_count = sandbox.sdk2.parameters.Integer("Number of async requests", default=50)

    def _tvm_secret(self):
        yav_secret = sandbox.sdk2.yav.Secret(self.Parameters.yav_secret_id)
        return yav_secret.data()["client_secret"]

    def on_create(self):
        if self.Requirements.tasks_resource is None:
            # This code does not seem to work when running a task locally with
            # --enable-taskbox. The if-else solves this.
            logging.info("task binary is not set, looking for latest suitable one")
            self.Requirements.tasks_resource = (
                sandbox.sdk2.service_resources.SandboxTasksBinary.find(
                    attrs={"target": "maps/MapsMasstransitMtmatrixTester"},
                    state=sandbox.common.types.resource.State.READY
                ).first()
            )
            logging.info("task binary: {}".format(self.Requirements.tasks_resource))
        else:
            logging.info("task binary is already set")

    def on_execute(self):
        logging.info("checking that tasks resource is properly set")
        logging.info("current tasks_resource: {}".format(
            self.Requirements.tasks_resource))
        if self.Requirements.tasks_resource is None:
            raise sandbox.common.errors.TaskFailure(
                "self.Requirements.tasks_resource is not set"
                " for MapsMasstransitMtmatrixTester")

        from maps.masstransit.matrix.shooter.lib import shooter

        tvm_config_path = "tvm-config.json"
        with open(tvm_config_path, "w") as tvm_config:
            tvm_config.write(_TVM_CONFIG)

        os.environ["TVMTOOL_LOCAL_AUTHTOKEN"] = _random_auth_token()
        os.environ["TVM_SECRET"] = self._tvm_secret()
        os.makedirs(_TVMTOOL_CACHE_DIR)

        tvmtool_path = _prepare_tvmtool()
        with sandbox.sdk2.helpers.ProcessLog(self, logger="tvmtool") as tvm_log:
            tvmtool_process = subprocess.Popen(
                [
                    tvmtool_path,
                    "-c", tvm_config_path,
                    "--port", str(_TVMTOOL_PORT),
                    "--cache-dir", _TVMTOOL_CACHE_DIR,
                ],
                stdout=tvm_log.stdout,
                stderr=tvm_log.stderr)
            logging.info("shooting masstransit matrix router in testing")
            shooter.shoot(
                thread_count=self.Parameters.thread_count,
                sync_request_count=self.Parameters.sync_request_count,
                async_request_count=self.Parameters.async_request_count,
                tvm_source_alias="maps-core-masstransit-testing-tools",
                tvm_target_alias="maps-core-masstransit-matrix",
                tvm_s3_alias="s3-mds",
                tvmtool_port=_TVMTOOL_PORT)
            tvmtool_process.terminate()

    def on_break(self, prev_status, status):
        self._send_status_to_juggler("CRIT", status)
        super(MapsMasstransitMtmatrixTester, self).on_break(prev_status, status)

    def on_success(self, prev_status):
        self._send_status_to_juggler("OK", "SUCCESS")
        super(MapsMasstransitMtmatrixTester, self).on_success(prev_status)

    def on_failure(self, prev_status):
        self._send_status_to_juggler("CRIT", "FAILURE")
        super(MapsMasstransitMtmatrixTester, self).on_failure(prev_status)
