from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
import sandbox.common.platform

import time
import json
import requests
import os
import subprocess
import logging
import binascii
from datetime import datetime

DEFAULT_YT_OUTPUT_TABLE = '//home/yatransport-prod/testing/mtroute-statistics/regression/history'
DEFAULT_YT_CLUSTER = 'hahn'
DEFAULT_ARCADIA_FILE_PATH = 'arcadia:arc/trunk/arcadia/maps/masstransit/router/regression/data/routes.json'
DEFAULT_ROUTER_URL = 'http://core-masstransit-router.common.testing.maps.yandex.net'
DEFAULT_DST_TVM_ALIAS = 'maps-core-masstransit-router-testing'
DEFAULT_DST_TVM_ID = '2012632'
YAV_SECRET_ID = 'sec-01f5jyk087p28t19ydezkqezx2'

MAX_NUMBER_OF_RETRIES = 5

DATE_STRING_COLUMN_NAME = 'date_string'
SUCCESS_COLUMN_NAME = 'success_requests_number'
TOTAL_COLUMN_NAME = 'total_requests_number'


class TvmClient:
    _TVMTOOL_PORT = 12346
    _TVMTOOL_CACHE_DIR = "tvmtool-cache-dir"
    _DEFAULT_TESTING_TOOLS_ALIAS = "maps-core-masstransit-testing-tools"

    @staticmethod
    def make_config(dst_alias, dst_id):
        config = \
            """
                {
                    "clients": {
            """ \
            + '"' + TvmClient._DEFAULT_TESTING_TOOLS_ALIAS + '": {' + \
            """
                        "secret": "env:TVM_SECRET",
                        "self_tvm_id": 2028046,
                        "dsts": {
            """ \
            + '"' + dst_alias + '": { "dst_id": ' + dst_id + ' }' + \
            """
                        }
                    }
                }
            }
            """
        logging.info('TVM tool config:\n' + config)
        return config

    def __init__(self, task, dst_tvm_alias, dst_tvm_id):
        tvm_config_path = "tvm-config.json"
        with open(tvm_config_path, "w") as tvm_config:
            tvm_config.write(self.make_config(dst_tvm_alias, dst_tvm_id))

        os.environ["TVMTOOL_LOCAL_AUTHTOKEN"] = TvmClient._random_auth_token()
        os.environ["TVM_SECRET"] = self._tvm_secret()
        os.makedirs(self._TVMTOOL_CACHE_DIR)

        tvmtool_path = TvmClient._prepare_tvmtool()

        self._tvm_log = sdk2.helpers.ProcessLog(
            task=task,
            logger="tvmtool"
        )
        self._tvmtool_process = subprocess.Popen(
            [
                tvmtool_path,
                "-c", tvm_config_path,
                "--port", str(self._TVMTOOL_PORT),
                "--cache-dir", self._TVMTOOL_CACHE_DIR,
            ],
            stdout=self._tvm_log.stdout,
            stderr=self._tvm_log.stderr
        )
        from maps.routing.matrix_router.quality_reporter.shooter.lib.shooter import \
            make_tvm_client
        self._tvm_client = make_tvm_client(
            tvmtool_port=self._TVMTOOL_PORT,
            auth_token=os.environ["TVMTOOL_LOCAL_AUTHTOKEN"],
            source_alias=self._DEFAULT_TESTING_TOOLS_ALIAS
        )
        self._dst_tvm_id = dst_tvm_id

    def _tvm_secret(self):
        yav_secret = sdk2.yav.Secret(YAV_SECRET_ID)
        return yav_secret.data()["client_secret"]

    def get_ticket(self):
        return self._tvm_client.get_service_ticket_for(tvm_id=self._dst_tvm_id)

    def stop(self):
        self._tvmtool_process.terminate()
        self._tvm_log.close()

    @staticmethod
    def _prepare_tvmtool():
        platform = sandbox.common.platform.platform()
        arch = sandbox.common.platform.get_arch_from_platform(platform)
        tvmtool_resource = sdk2.Resource.find(
            type="TVM_TOOL_BINARY", arch=arch).first()
        return str(sdk2.ResourceData(tvmtool_resource).path)

    @staticmethod
    def _random_auth_token():
        """Generate a random string to be used as a TVM authorization token.

        More on this token:
        https://wiki.yandex-team.ru/passport/tvm2/tvm-daemon/#kakzapustitprocess
        """
        return binascii.hexlify(os.urandom(16)).decode("utf-8")


class RegressionMeter:
    def __init__(self, router_url, requests_file, tvm_client):
        with open(requests_file, 'r') as f:
            self.requests_list = json.load(f)
        self.router_url = router_url
        self.success_number = 0
        self._tvm_client = tvm_client

    @staticmethod
    def check_response(expected_response, actual_response):
        from maps.masstransit.statistics.router.pylibs.common_py3.lib.calc_regression import validate_response
        return validate_response(expected_response, actual_response)

    def measure(self):
        from maps.masstransit.statistics.router.pylibs.common_py3.lib.calc_regression import generate_request_url
        from retry import retry

        @retry(tries=MAX_NUMBER_OF_RETRIES, delay=1)
        def make_request(url):
            ticket = self._tvm_client.get_ticket()
            response = requests.get(
                url,
                timeout=10,
                headers={'X-Ya-Service-Ticket': ticket}
            )
            response.raise_for_status()
            return response.content

        bad_requests = []
        for request_entry in self.requests_list:
            full_request_url = self.router_url + generate_request_url(request_entry["request"])
            response = make_request(full_request_url)
            if self.check_response(request_entry["expected_response"], response):
                self.success_number += 1
            else:
                bad_requests.append(request_entry)
            time.sleep(0.1)

        return (self.success_number, len(self.requests_list), bad_requests)


class MapsMasstransitMtrouterQualityRegressionArtifact(sdk2.Resource):
    """JSON file with bad responsed requests"""


class MapsMasstransitMtrouterQualityRegression(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 1024  # MB
        ram = 1024  # MB

    class Parameters(sdk2.Task.Parameters):
        cluster = sdk2.parameters.String(
            'YT cluster',
            default=DEFAULT_YT_CLUSTER
        )
        output_table = sdk2.parameters.String(
            'Path to the YT output table',
            default=DEFAULT_YT_OUTPUT_TABLE
        )
        arcadia_file_path = sdk2.parameters.String(
            'Path to file with requests list in arcadia',
            default=DEFAULT_ARCADIA_FILE_PATH
        )
        router_url = sdk2.parameters.String(
            'Router URL',
            default=DEFAULT_ROUTER_URL
        )
        dst_tvm_alias = sdk2.parameters.String(
            'Destination TVM alias',
            default=DEFAULT_DST_TVM_ALIAS
        )
        dst_tvm_id = sdk2.parameters.String(
            'Destination TVM id',
            default=DEFAULT_DST_TVM_ID
        )

    def _write_to_yt(self, success_number, total_number):
        import yt.wrapper as yt
        yt.config.set_proxy(self.Parameters.cluster)
        yt.config['token'] = sdk2.Vault.data('robot_robot-mtr_yt_token')
        with yt.Transaction(timeout=5*60*1000):  # 5 minutes
            if not yt.exists(self.Parameters.output_table):
                yt_table_directory = yt.ypath_dirname(self.Parameters.output_table)
                yt.mkdir(yt_table_directory)
                yt.create_table(
                    self.Parameters.output_table,
                    attributes={"schema": [
                        {"name": DATE_STRING_COLUMN_NAME, "type": "string", "sort_order": "ascending"},
                        {"name": SUCCESS_COLUMN_NAME, "type": "int64"},
                        {"name": TOTAL_COLUMN_NAME, "type": "int64"}
                    ]}
                )
            current_date_str = datetime.strftime(datetime.now(), "%Y-%m-%d %H:%M:%S")
            row = {
                DATE_STRING_COLUMN_NAME: current_date_str,
                SUCCESS_COLUMN_NAME: success_number,
                TOTAL_COLUMN_NAME: total_number
            }
            yt.write_table(yt.TablePath(self.Parameters.output_table, append=True), [row])

    def on_execute(self):
        logging.basicConfig()
        requests_file = "requests.json"
        Arcadia.export(self.Parameters.arcadia_file_path, requests_file)

        tvm_client = TvmClient(
            self,
            self.Parameters.dst_tvm_alias,
            self.Parameters.dst_tvm_id
        )
        regression_meter = RegressionMeter(
            self.Parameters.router_url,
            requests_file,
            tvm_client
        )
        success_number, total_number, bad_requests = regression_meter.measure()
        tvm_client.stop()

        resource = MapsMasstransitMtrouterQualityRegressionArtifact(
            self,
            "Mtroute regression artifact",
            "failed.json"
        )
        resource_data = sdk2.ResourceData(resource)
        with open(str(resource_data.path), 'w') as output_file:
            json.dump(bad_requests, output_file, indent=4)
        resource_data.ready()

        self._write_to_yt(success_number, total_number)
