from sandbox import sdk2
from sandbox.projects.masstransit.common.shooting import make_simple_load_config, make_step_rps
from sandbox.projects.masstransit.common.subtasks import ParentTask, SubtasksError
from sandbox.projects.masstransit.common.utils import make_shooting_stage
from sandbox.projects.masstransit.MapsMasstransitPrepareProductionLogAmmo import MapsMasstransitPrepareProductionLogAmmo
from sandbox.projects.masstransit.MapsMasstransitRestartService import MapsMasstransitRestartService
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi

AMMO_RESOURCE_TYPE='MAPS_MASSTRANSIT_PRODUCTION_LOG_AMMO'
# TODO: update vhost
VHOST='mt-router.maps.yandex.net'


class MapsMasstransitMtrouterShooting(ParentTask):

    class Parameters(sdk2.Parameters):
        restart_service = sdk2.parameters.Bool('Restart service', default=True)

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def generate_mtrouter_ammo(self, handle, requests_number=1000000):
        stage = 'Ammo generator for {0}'.format(handle)
        ammo_task = self.run_subtask(
            stage,
            lambda: MapsMasstransitPrepareProductionLogAmmo(
                self,
                description=stage,
                notifications=self.Parameters.notifications,
                resource_type=AMMO_RESOURCE_TYPE,
                vhost=VHOST,
                requests_number=requests_number,
                handles=[handle]
            )
        )
        return ammo_task.Parameters.ammo

    def shoot_mtrouter(self, handle, rps, ammo_resource, autostop, shooting_name_suffix):
        headers = [
            'Host: mt-router.maps.yandex.net',
            'Accept: application/x-protobuf',
            'X-Ya-Service-Ticket: {0}'.format(sdk2.Vault.data(
                'maps-core-masstransit-router-load-tvm-ticket')
            ),
        ]
        config_content = make_simple_load_config(
            task='MTDEV-487',
            component=handle,
            rps=rps,
            headers=headers,
            autostop=autostop,
            shooting_name_suffix=shooting_name_suffix
        )
        stage = make_shooting_stage(handle, rps, shooting_name_suffix)

        self.run_subtask(
            stage,
            lambda: ShootViaTankapi(
                self,
                description='Shooting {0} with rps {1}'.format(handle, rps),
                notifications=self.Parameters.notifications,
                use_public_tanks=False,
                tanks=['nanny:maps_core_tanks_load'],
                nanny_service='maps_core_masstransit_router_load',
                config_source='file',
                config_content=config_content,
                use_monitoring=True,
                monitoring_source='arcadia',
                monitoring_arc_path='maps/masstransit/router/regression/telegraf_config.cfg',
                ammo_source='resource',
                ammo_resource=ammo_resource
            )
        )

    def run_load_test(self, handle, mtrouter_rps, ammo_resource='', autostop=[],
                      shooting_name_suffix=None):
        try:
            if ammo_resource:
                mtrouter_ammo_resource = ammo_resource
            else:
                mtrouter_ammo_resource = self.generate_mtrouter_ammo(handle)

            for rps in mtrouter_rps:
                try:
                    self.shoot_mtrouter(
                        handle,
                        rps,
                        mtrouter_ammo_resource,
                        autostop,
                        shooting_name_suffix
                    )
                except SubtasksError:
                    pass
        except SubtasksError:
            # Do not fail the parent task, continue with next load tests
            pass

    def restart_mtrouter(self):
        self.run_subtask(
            'Restart MtRouter',
            lambda: MapsMasstransitRestartService(
                self,
                description='Restarting MtRouter before load tests',
                nanny_service_id='maps_core_masstransit_router_load'
            )
        )

    def on_execute(self):
        if self.Parameters.restart_service:
            self.restart_mtrouter()

        self.run_load_test(
            handle='/masstransit/v2/isochrone',
            mtrouter_rps=[
                make_step_rps(16)
            ]
        )

        self.run_load_test(
            handle='/masstransit/v2/route',
            mtrouter_rps=[
                make_step_rps(max_rps=260, min_rps=110)
            ]
        )

        self.run_load_test(
            handle='/masstransit/v2/summary',
            mtrouter_rps=[
                make_step_rps(500)
            ]
        )

        self.run_load_test(
            handle='/masstransit/v2/uri',
            mtrouter_rps=[
                make_step_rps(max_rps=6500, min_rps=3500)
            ],
            autostop=[
                'quantile(50,100,20)',
                'http(5xx,10%,3)',
                'http(4xx,25%,4)',
                'net(101,25,5)'
            ]
        )

        self.run_load_test(
            handle='/pedestrian/v2/route',
            mtrouter_rps=[
                make_step_rps(max_rps=7000, min_rps=4000)
            ],
            autostop=[
                'quantile(50,100,20)',
                'http(5xx,10%,3)',
                'http(4xx,25%,4)',
                'net(101,25,5)'
            ]
        )

        self.run_load_test(
            handle='/pedestrian/v2/summary',
            mtrouter_rps=[
                make_step_rps(max_rps=12000, min_rps=6000)
            ],
            autostop=[
                'quantile(99,100,20)',
                'http(5xx,10%,3)',
                'http(4xx,25%,4)',
                'net(101,25,5)'
            ]
        )

        self.run_load_test(
            handle='/pedestrian/v2/matrix',
            mtrouter_rps=[
                make_step_rps(max_rps=2000, min_rps=800)
            ]
        )

        self.run_load_test(
            handle='/pedestrian/v2/matrix',
            mtrouter_rps=[
                make_step_rps(2500)
            ],
            ammo_resource='1586702438',
            shooting_name_suffix='random matrices 1x25'
        )
        self.run_load_test(
            handle='/pedestrian/v2/matrix',
            mtrouter_rps=[
                make_step_rps(1200)
            ],
            ammo_resource='1586834256',
            shooting_name_suffix='random matrices 1x50'
        )
        self.run_load_test(
            handle='/pedestrian/v2/matrix',
            mtrouter_rps=[
                make_step_rps(800)
            ],
            ammo_resource='1586938453',
            shooting_name_suffix='random matrices 1x100'
        )

        self.run_load_test(
            handle='/pedestrian/v2/uri',
            mtrouter_rps=[
                make_step_rps(2600)
            ]
        )

        self.run_load_test(
            handle='/pedestrian/v2/isochrone',
            mtrouter_rps=[
                make_step_rps(max_rps=2000, min_rps=500)
            ]
        )
