from collections import namedtuple
import datetime

from sandbox import common, sdk2
from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin
from sandbox.sdk2.vcs.svn import Arcadia


URL_TEMPLATE_MTINFO_REGIONS = "arcadia:/arc/trunk/arcadia/maps/masstransit/configs/mtinfo_regions/{0}"
URL_REGIONS_JSON = "arcadia:/arc/trunk/arcadia/maps/masstransit/configs/regions-json/regions.json"
MTINFO_REGIONS_NAME_IN_DATASET = "masstransit-regions.conf"
REGIONS_JSON_NAME_IN_DATASET = "regions.json"
YT_PATH_PREFIX_TEMPLATE = "//home/maps/jams/{env}/masstransit/regions_config"

Config = namedtuple("Config", [
    "mtinfo_regions_filename", "dataset", "ecstatic_env",
    "tvm_id", "tvm_secret_id"
])
ENVIRONMENT_CONFIGS = {
    "testing": Config(
        mtinfo_regions_filename="masstransit-regions.conf.default",
        dataset="yandex-maps-masstransit-regions-config",
        ecstatic_env="testing",
        tvm_id="2028046",  # maps-core-masstransit
        tvm_secret_id="sec-01f5jyk087p28t19ydezkqezx2",
    ),
    "datatesting": Config(
        mtinfo_regions_filename="masstransit-regions.conf.datatesting",
        dataset="yandex-maps-masstransit-regions-config-datatesting",
        ecstatic_env="datatesting",
        tvm_id="2028046",  # maps-core-masstransit
        tvm_secret_id="sec-01f5jyk087p28t19ydezkqezx2",
    ),
    "production": Config(
        mtinfo_regions_filename="masstransit-regions.conf.production",
        dataset="yandex-maps-masstransit-regions-config-production",
        ecstatic_env="stable",
        tvm_id="2020679",  # maps-core-masstransit
        tvm_secret_id="sec-01e9za2yfnh2e2rzs8f4yw33zz",
    ),
}


class MapsMasstransitUpdateRegionsConfig(MapsEcstaticToolMixin, sdk2.Task):
    """ Updates masstransit-regions.conf """

    class Requirements(sdk2.Requirements):
        disk_space = 2 * 1024  # 2GB
        cores = 1

        environments = (
            sdk2.environments.PipEnvironment("yandex-yt"),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):

        environment = sdk2.parameters.String(
            "Environment",
            choices=[(k, k) for k in ENVIRONMENT_CONFIGS.iterkeys()],
            default="testing",
            required=True)
        revision = sdk2.parameters.Integer("Revision", default=None)

    def on_execute(self):
        config = ENVIRONMENT_CONFIGS[self.Parameters.environment]

        data_dir = self.path("dataset_dir")
        common.fs.make_folder(data_dir)

        Arcadia.export(
            URL_TEMPLATE_MTINFO_REGIONS.format(config.mtinfo_regions_filename),
            path=data_dir / MTINFO_REGIONS_NAME_IN_DATASET,
            revision=self.Parameters.revision)

        Arcadia.export(
            URL_REGIONS_JSON,
            path=data_dir / REGIONS_JSON_NAME_IN_DATASET,
            revision=self.Parameters.revision)

        version = str(self.Parameters.revision)
        self.ecstatic(
            config.ecstatic_env,
            ["upload", config.dataset + "=" + version, str(data_dir), "+stable"],
            tvm_id=config.tvm_id,
            tvm_secret_id=config.tvm_secret_id,
        )

        if self.Parameters.environment in ["testing", "production"]:
            self.upload_file_to_yt(str(data_dir / MTINFO_REGIONS_NAME_IN_DATASET), version)

    def upload_file_to_yt(self, filepath, version):
        import yt.wrapper as yt  # noqa, it's forbidden to import yt globally in this type of SB-tasks

        client = yt.YtClient(proxy="hahn", token=sdk2.Vault.data("robot_robot-mtr_yt_token"))
        yt_path_prefix = YT_PATH_PREFIX_TEMPLATE.format(env=self.Parameters.environment)
        yt_path_with_version = yt.ypath_join(yt_path_prefix, version)
        yt_fullpath_to_file = yt.ypath_join(yt_path_with_version, MTINFO_REGIONS_NAME_IN_DATASET)

        with client.Transaction():
            client.create(
                "map_node",
                path=yt_path_with_version,
            )
            client.smart_upload_file(
                filename=filepath,
                destination=yt_fullpath_to_file,
                placement_strategy="replace",
            )
            expiration_time = datetime.datetime.isoformat(datetime.datetime.utcnow() + datetime.timedelta(days=30))
            for cfg in client.list(yt_path_prefix, absolute=True):
                cfg_exp_time_attr = yt.ypath_join(cfg, '@expiration_time')
                if cfg != yt_path_with_version and not client.exists(cfg_exp_time_attr):
                    client.set(cfg_exp_time_attr, expiration_time)
