from datetime import datetime

import sandbox.sandboxsdk.copy as sdk_copy
import sandbox.common.types.misc as ctm

from sandbox import sdk2

ENVIRONMENTS = ("stable", "datatesting")

class MapsMasstransitSrcExportResource(sdk2.Resource):
    version = sdk2.parameters.String("Resource version")
    environment = sdk2.parameters.String("Environment where the resource will be used")


class MapsMasstransitSrcExportDownloader(sdk2.Task):
    """Task for downloading masstransit data from rasp.yandex-team.ru."""

    RASP_TABLES = [
        'calendar',
        'frequency',
        'geometry',
        'l10n',
        'routes',
        'stops',
        'threads',
        'thread_stops',
        'timetable',
        'transitions',
    ]

    @staticmethod
    def _generate_version():
        timestamp = datetime.now()
        return timestamp.strftime('%Y.%m.%d.%H.%M')

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        environment = sdk2.parameters.String(
            "Environment",
            choices=[(k, k) for k in ENVIRONMENTS],
            default="datatesting"
        )

        rasp_url = sdk2.parameters.String(
            "Rasp url, should have {} in place of file name",
            default='http://rasp-bucket.s3.mds.yandex.net/rasp-long-haul/2.5/{}')
        version = sdk2.parameters.String(
            "Version (usually in format YYYY.mm.dd.HH.MM. If not specified, task start datetime will be used.")

    def on_execute(self):
        resource = MapsMasstransitSrcExportResource(
            self,
            "masstransit rasp export data",
            "rasp_export_src", ttl='inf')
        resource.environment = self.Parameters.environment
        resource.version = self.Parameters.version or self._generate_version()

        data = sdk2.ResourceData(resource)
        data.path.mkdir(0o755, parents=True, exist_ok=True)

        for table in self.RASP_TABLES:
            src_path = self.Parameters.rasp_url.format(table)
            dst_path = data.path.joinpath(table)
            # when input file is empty RemoteCopy don't create the output one
            dst_path.touch()
            sdk_copy.RemoteCopy(
                src_path, str(dst_path), log_dir=self.log_path())(
                timeout=sdk_copy.RemoteCopy.DEFAULT_TIMEOUT)
