import logging
from sandbox import sdk2

from sandbox.projects.masstransit.resources import (
    MapsMasstransitDataMms,
)
from sandbox.projects.masstransit.common.utils import (
    get_latest_resource,
    LatestResourceParameter
)

import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr


ENVIRONMENT_TO_RELEASE = {
    'development': ctt.ReleaseStatus.UNSTABLE,
    'testing': ctt.ReleaseStatus.TESTING,
    'datatesting': ctt.ReleaseStatus.STABLE,
    'production': ctt.ReleaseStatus.STABLE,
}


class MapsMasstransitToolsStageTimesBin(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-MASSTRANSIT']


def get_active_data_mms(environment):
    return get_latest_resource(
        MapsMasstransitDataMms,
        state=ctr.State.READY,
        attrs={'released': ENVIRONMENT_TO_RELEASE[environment]}
    )


class MapsMasstransitStageTimes(sdk2.Task):
    '''
    https://wiki.yandex-team.ru/users/likynushka/stagetimes
    '''

    class Parameters(sdk2.Task.Parameters):
        yt_cluster = sdk2.parameters.String(
            'YT cluster',
            default='hahn.yt.yandex.net'
        )

        environment = sdk2.parameters.String(
            'Environment where stage times will be calculated',
            default='development',
            choices=[
                (env, env) for env in ENVIRONMENT_TO_RELEASE
            ]
        )

        data_mms = sdk2.parameters.Resource(
            'data.mms. If not specified, most recently generated active data.mms is used',
            resource_type=MapsMasstransitDataMms
        )

        stage_times = LatestResourceParameter(
            'stage_times',
            resource_type=MapsMasstransitToolsStageTimesBin,
            required=True
        )

        output_path = sdk2.parameters.String(
            'YT path to the output directory. If not specified then deduced for selected environment'
        )

    def _run_stage_times(self, stage_times, data_mms):
        from maps.masstransit.statistics.router.tools.stage_times.lib.stage_times import (
            calculate_stage_times
        )
        from maps.masstransit.statistics.router.pylibs.common.lib.config import (
            STAGE_TIMES_TABLES
        )
        from maps.masstransit.pylibs.yt.lib.utils import dates_range
        from maps.masstransit.pylibs.yt.lib.config import Config
        import yt.wrapper as yt

        yt.config['token'] = sdk2.Vault.data('robot_robot-mtr_yt_token')
        yt.config.set_proxy(self.Parameters.yt_cluster)

        config = Config(self.Parameters.environment)
        output_path = self.Parameters.output_path or config.tables_root(
            STAGE_TIMES_TABLES
        )

        stage_times = sdk2.ResourceData(stage_times)
        data_mms = sdk2.ResourceData(data_mms)
        calculate_stage_times(
            config,
            output_path,
            str(data_mms.path),
            str(stage_times.path),
            dates_range()
        )

    def on_execute(self):
        logging.basicConfig(level=logging.INFO)
        data_mms = self.Parameters.data_mms or get_active_data_mms(self.Parameters.environment)

        self._run_stage_times(self.Parameters.stage_times, data_mms)
