#! /usr/bin/env python
# -*- coding: utf-8 -*-

import glob
import logging
import os
from sandbox import sdk2
import subprocess
import sys

import sandbox.common.types.client as Client
import sandbox.common.types.task as Task
from sandbox.projects.sandbox import resources


class MapsMasstransitTransportRegionsResource(sdk2.Resource):
    """ Regions config for mtinfo """
    releasable = True
    releasers = ["MAPS-MASSTRANSIT", "MASSTRANSIT_FEEDS"]
    arcadia_url = sdk2.parameters.String("Arcadia URL")
    arcadia_revision = sdk2.parameters.String("Arcadia revision")


class MapsMasstransitTransportRegions(sdk2.Task):
    ''' Upload transport regions to S3 bucket '''

    class Requirements(sdk2.task.Requirements):
        client_tags = Client.Tag.GENERIC
        disk_space = 512

    class Parameters(sdk2.Task.Parameters):
        description = 'Upload Transport Regions to S3 bucket'
        with sdk2.parameters.CheckGroup('Release files to') as auto_release:
            auto_release.values[Task.ReleaseStatus.TESTING] = auto_release.Value('Testing', checked=True)
            auto_release.values[Task.ReleaseStatus.STABLE] = auto_release.Value('Stable')
        arcadia_url = sdk2.parameters.String(
            'Arcadia URL',
            default='arcadia:/arc/trunk/arcadia/maps/masstransit/configs/regions-json',
            required=True,
        )
        s3_bucket = sdk2.parameters.String(
            'S3 bucket name',
            default='mtr-transport-regions',
            required=True,
        )
        s3_delete_extraneous = sdk2.parameters.Bool(
            'Delete extraneous files from S3 bucket',
            default=True,
        )

    def list_json_files(self, local_path):
        logging.info('Looking for JSON files in %s', local_path)
        json_files = glob.glob('{}/*.json'.format(local_path))
        if json_files:
            logging.info('%d JSON files found', len(json_files))
            return json_files
        else:
            raise Exception('Could not find *.json files in provided directory')

    def create_resource(self, arcadia_url):
        arcadia_revision = sdk2.vcs.svn.Arcadia.get_revision(arcadia_url)
        description = 'Transport Regions {id}-{revision}'.format(
            id=self.id,
            revision=arcadia_revision,
        )
        logging.info('Create new resource: %s', description)
        resource = MapsMasstransitTransportRegionsResource(
            self, description, 'regions',
            arcadia_url=arcadia_url,
            arcadia_revision=arcadia_revision,
        )
        resource_data = sdk2.ResourceData(resource)
        resource_path = str(resource_data.path)
        logging.info('Checkout SVN "%s" r%s', arcadia_url, arcadia_revision)
        sdk2.svn.Arcadia.export(arcadia_url, resource_path, revision=arcadia_revision)
        resource_data.ready()
        logging.info('Resource #%s is ready to release', resource.id)

    def get_resource_data(self, type, **kwargs):
        logging.info('Retrieving "{type}" resource'.format(type=str(type)))
        resource = type.find(**kwargs).order(-sdk2.Task.id).first()
        if resource:
            logging.info('Found suitable resource #%s', resource.id)
            return str(sdk2.ResourceData(resource).path)
        else:
            raise Exception('Cannot find suitalbe resource')

    def get_s3_credentials(self, release_to):
        vault_suffix = release_to.upper()
        vault_aws_key = 'MAPS_MT_AWS_KEY_{}'.format(vault_suffix)
        vault_aws_key_id = 'MAPS_MT_AWS_KEY_ID_{}'.format(vault_suffix)
        logging.info('Retrieving vault data for keys: %s, %s', vault_aws_key, vault_aws_key_id)
        return dict(
            AWS_SECRET_ACCESS_KEY=sdk2.Vault.data(vault_aws_key),
            AWS_ACCESS_KEY_ID=sdk2.Vault.data(vault_aws_key_id),
        )

    def upload_files(self, release_to):
        parameters = self.Parameters
        stable_release = release_to == Task.ReleaseStatus.STABLE
        endpoint = 'http://s3.mds.yandex.net' if stable_release else 'http://s3.mdst.yandex.net'
        s3_url = '{endpoint}/{parameters.s3_bucket}'.format(**locals())
        logging.info('Uploading Transport Regions to {s3_url}'.format(**locals()))

        aws = self.get_resource_data(
            resources.AWSCLISFX,
            attrs=dict(released='stable'),
        )
        local_path = self.get_resource_data(
            MapsMasstransitTransportRegionsResource,
            task_id=int(self),
        )
        self.list_json_files(local_path)

        s3_cred = self.get_s3_credentials(release_to)
        environ = os.environ.copy()
        environ.update(s3_cred)

        sync_cmd = [
            sys.executable, aws,
            '--endpoint-url', endpoint,
            's3', 'sync', local_path,
            's3://{bucket}'.format(bucket=parameters.s3_bucket),
            '--include', '*.json',
        ]
        if parameters.s3_delete_extraneous:
            sync_cmd.append('--delete')

        logging.info('Executing: {command}'.format(command=' '.join(sync_cmd)))
        with sdk2.helpers.ProcessLog(logger=logging.getLogger("aws")) as plog:
            retcode = subprocess.Popen(
                sync_cmd,
                env=environ,
                stdout=plog.stdout,
                stderr=subprocess.STDOUT,
            ).wait()
            if not retcode:
                logging.info('All files have been successfully uploaded to {s3_url}'.format(**locals()))
            else:
                raise Exception('Failed to upload files to {s3_url}'.format(**locals()))

    def on_execute(self):
        self.create_resource(self.Parameters.arcadia_url)
        for release_to in self.Parameters.auto_release:
            self.upload_files(release_to)

    def on_release(self, additional_parameters):
        release_to = additional_parameters['release_status']
        self.upload_files(release_to)
        sdk2.Task.on_release(self, additional_parameters)
