import logging
import urllib2
import json
import csv
from contextlib import closing
from datetime import datetime
from ftplib import FTP
from sandbox import sdk2
from sandbox.common.types.misc import DnsType
from sandbox.common.types.task import ReleaseStatus
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.common.nanny import nanny


ALLOWED_PROPERTIES = [
    'low_floor',
    'wheelchair_accessible',
    'bikes_allowed',
    'air_conditioning'
]

CSV_FORMAT = {
    'delimiter': '\t',
    'quoting': csv.QUOTE_NONE
}


def download_http(source):
    URL_TEMPLATE = 'http://{server}:{port}/{relative_url}'
    url = URL_TEMPLATE.format(**source)
    logging.info("Url: " + str(url))
    with closing(urllib2.urlopen(url, timeout=10)) as response:
        for line in response:
            yield line


def download_ftp(source):
    # We cannot use urllib2.urlopen for ftp, because it has a bug
    # It uses socket.gethostbyname instead of socket.getaddrinfo,
    # so it does not support ipv6 and dualstack hosts
    # More details: https://bugs.python.org/issue1675455
    ftp = FTP()
    ftp.connect(source['server'], source['port'])
    ftp.login(source['login'], source['password'])
    lines = []
    ftp.retrlines('RETR ' + source['relative_url'], lines.append)
    return lines


def download(source):
    DOWNLOADERS = {
        'http': download_http,
        'ftp': download_ftp,
    }
    return DOWNLOADERS[source['protocol']](source)


def validate(row):
    clid, vid, properties = row
    if not all(p in ALLOWED_PROPERTIES for p in properties.split(',')):
        raise ValueError("Invalid properties for {0}".format(row))


def process_source(source):
    lines = download(source)
    rows = iter(csv.reader(lines, **CSV_FORMAT))
    if source.get('skip_header_line', False):
        next(rows)
    for row in rows:
        validate(row)
        yield row


class MapsMasstransitVehicleProperties(sdk2.Resource):
    """ Vehicle properties for mtinfo """
    releasable = True
    releasers = ["MAPS-MASSTRANSIT"]


class MapsMasstransitUpdateVehicleProperties(nanny.ReleaseToNannyTask2, sdk2.Task):
    """ Updates masstransit vehicle properties """

    class Requirements(sdk2.Requirements):
        disk_space = 2 * 1024                      # 2GB
        cores = 1
        dns = DnsType.DNS64                         # Access to IPv4

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Url to sources config in Arcadia",
            default_value="arcadia:/arc/trunk/arcadia/maps/masstransit/configs/vehicle_properties/sources.json",
            required=True)

    def _read_config(self):
        path = "config"
        Arcadia.export(self.Parameters.arcadia_url, path)
        with open(path) as f:
            return json.load(f)

    def on_execute(self):
        sources = self._read_config()

        resource = sdk2.ResourceData(
            MapsMasstransitVehicleProperties(
                self, "Vehicle properties", "properties"))

        # resource.path.open uses unicode, but csv does not support it
        # Need str, because resource.path is PosixPath object
        with open(str(resource.path), 'w') as resource_file:
            writer = csv.writer(resource_file, **CSV_FORMAT)
            for source in sources['sources']:
                writer.writerows(process_source(source))

        resource.ready()

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)
        description = "Vehicle properties: {0}".format(datetime.now())

        # Oddly, but calling on_release from on_success is the recommended
        # way to update nanny services with new sandbox resources
        # https://wiki.yandex-team.ru/sandbox/cookbook/#kakavtomaticheskirelizitzadachu
        nanny.ReleaseToNannyTask2.on_release(self, dict(
            releaser=self.author,
            release_status=ReleaseStatus.STABLE,
            release_subject=description,
            release_comments=description,
            email_notifications=dict(to=[], cc=[]),
        ))
