from collections import namedtuple

from sandbox import sdk2
from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin
from sandbox.sdk2.vcs.svn import Arcadia


ARCADIA_PATH = "arcadia:/arc/trunk/arcadia/maps/masstransit/configs/route_blacklist/route-blacklist.json"
DATASET = "yandex-maps-masstransit-predictor-blacklist"

Config = namedtuple("Config", [
    "environment", "tvm_id", "tvm_secret_id"
])
TESTING_CONFIG = Config(
    environment="datatesting",
    tvm_id="2028046",  # maps-core-masstransit
    tvm_secret_id="sec-01f5jyk087p28t19ydezkqezx2"
)
STABLE_CONFIG = Config(
    environment="stable",
    tvm_id="2020679",  # maps-core-masstransit
    tvm_secret_id="sec-01e9za2yfnh2e2rzs8f4yw33zz"
)


class PredictorBlacklistJSON(sdk2.Resource):
    releasable = True
    releasers = ["MAPS-MASSTRANSIT", "MASSTRANSIT_FEEDS"]
    revision = sdk2.resource.Attributes.Integer("revision")


class MapsMasstransitUploadPredictorBlacklist(MapsEcstaticToolMixin, sdk2.Task):
    """ Updates route-blacklist.json """

    class Parameters(sdk2.Parameters):
        revision = sdk2.parameters.Integer("Revision", default=None, required=True)

    def on_execute(self):
        resource = PredictorBlacklistJSON(
            self,
            "Predictor blacklist: {}".format(self.Parameters.revision),
            "data")
        resource.revision = self.Parameters.revision
        resource_data = sdk2.ResourceData(resource)
        resource_data.path.mkdir(0755, parents=True, exist_ok=True)

        file_path = resource_data.path / "route-blacklist.json"
        Arcadia.export(ARCADIA_PATH, file_path, revision=self.Parameters.revision)

        resource_data.ready()
        self.upload(TESTING_CONFIG, resource_data.path)

    def on_release(self, params):
        resource = PredictorBlacklistJSON.find(
            attrs={"revision": self.Parameters.revision}
        ).first()
        self.upload(STABLE_CONFIG, sdk2.ResourceData(resource).path)
        sdk2.Task.on_release(self, params)
        self.mark_released_resources(params["release_status"], ttl=1)

    def upload(self, config, path):
        version = str(self.Parameters.revision)
        self.ecstatic(
            config.environment,
            ["upload", DATASET + "=" + version, str(path), "+stable"],
            tvm_id=config.tvm_id,
            tvm_secret_id=config.tvm_secret_id,
        )
