import re
import json
import math


class Target(object):
    def __init__(self, address, rps_schedule, ammo_resource, headers=[]):
        self.address = address
        self.rps_schedule = rps_schedule
        self.ammofile = ammo_resource.http_proxy
        self.headers = headers

    def config(self):
        return {
            'load_profile': {
                'load_type': 'rps',
                'schedule': ' '.join(self.rps_schedule),
            },
            'address': self.address,
            'headers': self.headers,
            'ammofile': self.ammofile,
            'instances': 10000,
        }


def set_autostop(config, autostop):
    config['autostop'] = {
        'autostop': autostop
    }


def set_default_autostop(config):
    config['autostop'] = {
        'autostop': [
            'quantile(50,200,20)',
            'http(5xx,10%,3)',
            'http(4xx,25%,4)',
            'net(101,25,5)'
        ]
    }


def set_rcassert(config, rcassert=[]):
    if not rcassert:
        # Set acceptable exit codes of tank
        # https://yandextank.readthedocs.io/en/latest/core_and_modules.html#basic-criteria-types
        # https://yandextank.readthedocs.io/en/latest/core_and_modules.html#exit-codes
        rcassert = {'pass': '0 21 22'}
    config['rcassert'] = rcassert


def make_load_config(
    task,
    component,
    targets,
    is_const=False,
    autostop=[]
):
    assert targets, 'At least one target required'

    config = {
        'phantom': targets[0].config(),
        'uploader': {
            'operator': 'lunapark',
            'task': task,
            'job_name': component,
            'component': component,
        },
    }

    if len(targets) > 1:
        config['phantom']['multi'] = [
            target.config() for target in targets[1:]
        ]

    # It's odd, but there should be 'autostop' inside 'autostop' here
    if autostop:
        set_autostop(config, autostop)

    # Non-const shoot is expected to overload service and fail by autostop
    # So 'fail' exit codes are actually considered 'pass'. See:
    #   https://yandextank.readthedocs.io/en/latest/config_reference.html#rcassert
    #   https://yandextank.readthedocs.io/en/latest/core_and_modules.html#exit-codes
    if not is_const:
        set_rcassert(config)

    # json is a correct yaml file, so we can use json here
    return json.dumps(config, indent=4)


def make_simple_load_config(
    task,
    component,
    rps,
    headers,
    autostop=[],
    shooting_name_suffix=None,
    ammo_type='uri'
):
    config = {
        'phantom': {
            'ammo_type': ammo_type,
            'enabled': True,
            'load_profile': {
                'load_type': 'rps',
                'schedule': rps
            },
            'header_http': '1.1',
            'headers': headers,
            'port': '80'
        },
        'uploader': {
            'operator': 'lunapark',
            'task': task,
            'job_name': component,
            'component': component
        }
    }

    if autostop:
        set_autostop(config, autostop)
    else:
        set_default_autostop(config)

    if 'const' in rps:
        config['uploader']['component'] += ' const'
    elif 'step' in rps:
        config['uploader']['component'] += ' step'

    if shooting_name_suffix is not None:
        config['uploader']['component'] += ' ' + shooting_name_suffix

    if 'const' not in rps:
        set_rcassert(config)

    return json.dumps(config, indent=4)


def make_const_rps(rps):
    # const(100, 10m) -> line(1,100,1m) const(100, 10m) or
    # const(5, 10m) -> line(0.5,5,1m) const(5, 10m)
    # to warm up the target before shooting
    line = 'line({},{},1m)'
    shooting_rps = float(re.findall(r'(\d+(?:\.\d+)?)', rps)[0])
    return ' '.join([line.format(min(shooting_rps / 10.0, 1), shooting_rps), rps])


def make_step_rps(max_rps, min_rps=None, rps_step_size=None, step_duration='2m', steps_number=10):
    steps_number = float(steps_number)
    if min_rps is None:
        min_rps = int(max_rps / 2.)
    if rps_step_size is None:
        rps_step_size = int(math.ceil((max_rps - min_rps) / steps_number))
    return "line(1,{min_rps},5m) step({min_rps},{},{},{})".format(max_rps, rps_step_size, step_duration, min_rps=min_rps)
