import logging
from datetime import date, timedelta, datetime
from sandbox import sdk2
from sandbox.common.patterns import classproperty
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.masstransit.resources import MASSTRANSIT_DATA

import copy
import tarfile
import tempfile


class DateParameter(sdk2.parameters.String):
    FORMAT = '%Y-%m-%d'

    @classmethod
    def cast(cls, value):
        try:
            datetime.strptime(value, cls.FORMAT)
        except ValueError:
            logging.error(
                "Invalid format of date parameter: "
                "must be YY-mm-dd, but equals {0}".format(value)
            )
            raise
        return value

    @staticmethod
    def yesterday():
        return date.strftime(date.today() - timedelta(days=1), DateParameter.FORMAT)

    @staticmethod
    def day_before_yesterday():
        return date.strftime(date.today() - timedelta(days=2), DateParameter.FORMAT)


class LatestResourceParameter(sdk2.parameters.Resource):
    """
    Class which sets the default parameter value to the latest
    available resource
    """
    @classproperty
    def default_value(cls):
        resources = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            state=ctr.State.READY,
            limit=1
        )['items']
        if not resources:
            return None
        return resources[0]['id']


def get_latest_resource(resource_type, **kwargs):
    return sdk2.Resource.find(
        resource_type=resource_type,
        **kwargs
    ).order(-sdk2.Resource.id).first()


def get_latest_stable_resource(resource_type, attrs={}):
    attrs = copy.deepcopy(attrs)
    attrs.update({"released": ctt.ReleaseStatus.STABLE})
    return get_latest_resource(resource_type, state=ctr.State.READY, attrs=attrs)


def get_stable_masstransit_data_resource():
    return get_latest_stable_resource(
        MASSTRANSIT_DATA,
        {"garden_environment": "production"})


def unpack_tzdata(tar_path):
    unpacked_path = tempfile.mkdtemp()
    with tarfile.open(tar_path) as archive:
        archive.extractall(unpacked_path)
    return unpacked_path


def make_shooting_stage(handle, rps, shooting_name_suffix=None):
    stage = handle
    if 'const' in rps:
        stage += ' const'
    if 'step' in rps:
        stage += ' step'
    if shooting_name_suffix:
        stage += ' ' + shooting_name_suffix
    return stage


def determine_nanny_hosts(
        nanny_service_id, nanny_api_url="https://nanny.yandex-team.ru",
        secrets_owner="MAPS-MASSTRANSIT", nanny_token_name="nanny_oauth_token"):
    nanny_client = NannyClient(
        api_url=nanny_api_url,
        oauth_token=sdk2.Vault.data(secrets_owner, nanny_token_name)
    )
    instances_info = nanny_client.get_service_current_instances(nanny_service_id)
    return [x["container_hostname"] for x in instances_info["result"]]
