import json
import logging
from sandbox import sdk2

from sandbox.common.utils import Enum

import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr

from sandbox.projects.masstransit.updateGTFS.MapsMasstransitUpdateGtfsSource import (
    MapsMasstransitGtfsSource,
    MapsMasstransitUpdateGtfsSource
)
from sandbox.projects.masstransit.updateGTFS.MapsMasstransitUpdateGtfsSourceWithAuth import (
    MapsMasstransitUpdateGtfsSourceWithAuth
)
from sandbox.projects.masstransit.common.utils import get_latest_resource

from .config import FEEDS, FEEDS_WITH_AUTH


class SearchFor(Enum):
    ANY_RESOURCE = 'Any resource, it can be not released'
    STABLE = 'Released to stable'
    TESTING = 'Released to testing'
    TESTING_OR_STABLE = 'Released to testing or stable'
    NONE = 'Forced download new resource'


def get_release_priority(release_type):
    try:
        return list(ctt.ReleaseStatus).index(release_type)
    except ValueError:
        return None


def search_last_resource(search_for, attrs):
    if search_for == SearchFor.NONE:
        return None
    elif search_for == SearchFor.TESTING:
        attrs['released'] = 'testing'
    elif search_for == SearchFor.STABLE:
        attrs['released'] = 'stable'
    elif search_for == SearchFor.TESTING_OR_STABLE:
        attrs['released'] = 'testing'
        testing_resource = get_latest_resource(MapsMasstransitGtfsSource, attrs=attrs, state='READY')
        attrs['released'] = 'stable'
        stable_resource = get_latest_resource(MapsMasstransitGtfsSource, attrs=attrs, state='READY')

        if testing_resource is not None and stable_resource is not None:
            if testing_resource.created > stable_resource.created:
                return testing_resource
            else:
                return stable_resource

        return stable_resource or testing_resource

    return get_latest_resource(MapsMasstransitGtfsSource, attrs=attrs, state='READY')


class MapsMasstransitAllFeedsUpdater(sdk2.Task):
    """
        Task for updating all gtfs sources
    """
    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 600  # 10 min
        with sdk2.parameters.String('Release resources to ') as release_type:
            release_type.values[''] = release_type.Value(value='do not release', default=True)
            for val in ctt.ReleaseStatus:
                release_type.values[val] = release_type.Value(value=val)
        search_for = sdk2.parameters.String('Resource search for ', default=SearchFor.NONE)
        search_for.choices = [(x, x) for x in SearchFor]
        with sdk2.parameters.Group("Info about cities for task MapsMasstransitUpdateGtfsSource") as group:
            cities = sdk2.parameters.Dict('The key of dict is a city, value of the dict is url', default=FEEDS)
        with sdk2.parameters.Group("Info about cities for task MapsMasstransitUpdateSourceFromEasyway") as group_easy:
            cities_with_auth = sdk2.parameters.Dict(
                'The key of dict is a city, value of the dict is url',
                default=FEEDS_WITH_AUTH)

    class Context(sdk2.Task.Context):
        wait_tasks = False

    def _create_subtask(self, subtask_type, city, url, cluster=None, auth_settings={}):
        logging.info('Create subtask for %s', city)
        resource = search_last_resource(self.Parameters.search_for, {'city': city})
        priority = ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.NORMAL)
        subtask = subtask_type(
            self,
            priority=priority,
            description='Subtask for downloading gtfs source {}'.format(city)
        )
        subtask.Parameters.compare_with = resource
        subtask.Parameters.city = city
        subtask.Parameters.url = url
        if cluster:
            subtask.Parameters.cluster = cluster
        if auth_settings:
            subtask.Parameters.login = auth_settings["login"]
            subtask.Parameters.vault_key = "robot-gtfs-{key}".format(key=auth_settings["vault_key"])
        subtask.save()
        return subtask

    def _spawn_subtasks(self):
        """
            Spawns subtasks that download gtfs for each city from parameters
        """
        logging.info('Begin indexing')
        children = []
        for city, url in self.Parameters.cities.items():
            subtask = self._create_subtask(MapsMasstransitUpdateGtfsSource, city, url)
            children.append(subtask)
        for city, info in self.Parameters.cities_with_auth.items():
            try:
                info = json.loads(info.replace("'", '"'))
                url = info['url']
                login = info['login']
                vault_key = info['vault_key']
            except Exception as e:
                logging.error("Cannot get auth parameters")
                raise e
            auth_settings = {
                'login': login,
                'vault_key': vault_key
            }
            subtask = self._create_subtask(
                subtask_type=MapsMasstransitUpdateGtfsSourceWithAuth,
                city=city,
                url=url,
                auth_settings=auth_settings
            )
            children.append(subtask)
        logging.info('Waiting subtasks to finish')
        self.Context.wait_tasks = True
        self.Context.save()
        raise sdk2.WaitTask([c.enqueue() for c in children], list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK), True)

    def _needs_release(self, resource):
        if not resource.type.releasable:
            return False
        task_release_priority = get_release_priority(resource.released)
        # Index of the best status is less than the worst one
        # because order ctt.ReleaseStatus is in descending order of priority
        if task_release_priority is not None and self.release_priority > task_release_priority:
            logging.info('Releasing priority of resource ({subtask}) is higher than {this_task}.'.format(
                subtask=resource.released,
                this_task=self.Parameters.release_type))
            return False
        logging.info('Releasing priority of resource ({subtask}) is lower than {this_task}.'.format(
            subtask=resource.released,
            this_task=self.Parameters.release_type))
        return True

    def _release_resource(self, resource, subject):
        logging.info('Releasing resource ({resource_id}) to {release_status}'.format(
            resource_id=resource.id,
            release_status=self.Parameters.release_type))
        self.server.release(
            task_id=resource.task_id,
            type=self.Parameters.release_type,
            subject=subject
        )

    def _release_tasks(self):
        for subtask in self.find(status=ctt.Status.Group.FINISH):
            subtask_resource = sdk2.Resource.find(
                MapsMasstransitGtfsSource,
                task=subtask,
                state=ctr.State.READY).first() if subtask.status == ctt.Status.SUCCESS else None
            compare_resource = subtask.Parameters.compare_with
            for resource in [subtask_resource, compare_resource]:
                if resource and self._needs_release(resource):
                    self._release_resource(resource, subtask.Parameters.city)
                    break

    def on_execute(self):
        if not self.Context.wait_tasks:
            self._spawn_subtasks()
        self.release_priority = get_release_priority(self.Parameters.release_type)
        if self.release_priority is not None:
            self._release_tasks()
