import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

import logging
from datetime import datetime
from email.utils import parsedate_tz, mktime_tz
from sandbox import sdk2

import sandbox.common.types.misc as ctm

REQUESTS_TIMEOUT = 10
CHUNK_SIZE = 1024


class MapsMasstransitGtfsSource(sdk2.Resource):
    """
        Public transportation source data
    """

    releasable = True
    auto_backup = True
    releasers = ["MAPS-MASSTRANSIT", "MASSTRANSIT_FEEDS"]
    city = sdk2.parameters.String(
        "City of resource",
        required=True
    )


def requests_retry_session(
    retries=8,
    backoff_factor=1,
    session=None,
):
    session = session or requests.Session()
    retry = Retry(
        total=retries,
        read=retries,
        connect=retries,
        backoff_factor=backoff_factor,
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session


class MapsMasstransitUpdateGtfsSource(sdk2.Task):
    """
    Base Task for Updating Gtfs Source
    """

    class Parameters(sdk2.Parameters):
        kill_timeout = 600  # 10 min
        with sdk2.parameters.Group("Info about city") as group:
            compare_with = sdk2.parameters.Resource(
                'Last resource for compare',
                resource_type=MapsMasstransitGtfsSource,
                default=None,
            )
            city = sdk2.parameters.String("City of data", required=True)
            url = sdk2.parameters.Url("Link for download data", required=True)

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def _get_date(self, str_date):
        date_tuple = parsedate_tz(str_date)
        timestamp = mktime_tz(date_tuple)
        return datetime.fromtimestamp(timestamp)

    def create_session(self):
        """
            if you need use auth, override this method
        """
        return None

    def _get_last_modified(self, url):
        s = self.create_session()
        headers = requests_retry_session(session=s).head(url, allow_redirects=True, timeout=REQUESTS_TIMEOUT)
        modified_date = headers.headers.get('Last-Modified')
        if modified_date is None:
            logging.warn('Cannot get Last-Modified header for url %s. '
                         'Use current date and time', url)
            return datetime.utcnow()
        return self._get_date(modified_date)

    def _load_source(self, resource):
        logging.info('Downloading file %s ...', self.Parameters.url)
        s = self.create_session()
        response = requests_retry_session(session=s).get(
            self.Parameters.url,
            allow_redirects=True,
            timeout=REQUESTS_TIMEOUT,
            stream=True
        )
        response.raise_for_status()
        resource_data = sdk2.ResourceData(resource)
        with resource_data.path.open('wb') as f:
            for chunk in response.iter_content(CHUNK_SIZE):
                f.write(chunk)
        resource_data.ready()

    def _make_resource(self):
        return MapsMasstransitGtfsSource(
            self,
            'Copy source from {}'.format(self.Parameters.url),
            'gtfs.zip',
            city=self.Parameters.city,
            ttl=40
        )

    def on_execute(self):
        city = self.Parameters.city
        latest_resource = self.Parameters.compare_with
        current_source_date = self._get_last_modified(self.Parameters.url).isoformat()
        if not latest_resource:
            logging.info('Upload first source data for %s', city)
            self._load_source(self._make_resource())
            return
        latest_uploaded_date = latest_resource.created.isoformat()
        if latest_uploaded_date < current_source_date:
            logging.info('Current source data for %s is new. Upload it to sandbox', city)
            self._load_source(self._make_resource())
            return

        logging.info('Sources for %s in sandbox are up to date', city)
