from sandbox import sdk2
from sandbox.common import errors as ce
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.projects.common import file_utils as fu

from sandbox.projects.tank.load_resources.resources import AMMO_FILE
from sandbox.projects.tank.load_resources.resources import YANDEX_TANKAPI_LXC_CONTAINER
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi

import yaml
import logging
import json
import os.path
import requests
import subprocess
import time
import ast
import os

DefaultProfile = {"100kb_get": 4, "10kb_get": 7, "1kb_get": 15, "10b_get": 24}


class S3Shooting(sdk2.Task):
    """ Shooting to S3 mds storage """

    class Context(sdk2.Task.Context):
        ammo_url = ""
        ammo_command = ""
        shooting_task_id = ""
        shooting_find_id = ""
        lxc_container = ""

    class Requirements(sdk2.Requirements):
        disk_space = 1024   # 1GiB on disk
        cores = 1           # exactly 1 core
        ram = 2048          # 2GiB or less

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Shooting parameters') as tankapi_block:
            task = sdk2.parameters.String('Lead ticket', default='LOAD-915')
            comment = sdk2.parameters.String('Comment on the shooting', default='Sandbox shooting')
            host = sdk2.parameters.String('Target host', default='storage.cloud-preprod.yandex.net:80', required=True)
            scheduler = sdk2.parameters.List('Load profile', default=['{"duration": "300s", "type": "const", "ops": 150}'])
            autostop = sdk2.parameters.List('Autostop requirements', default=['quantile(90,500ms,20s)', 'negative_http(2xx,20%,20s)'])
            instances = sdk2.parameters.Integer('Pandora instances', default=2000, required=True)

        with sdk2.parameters.Group('Ammo parameters') as ammo_block:
            ts = sdk2.parameters.String('Timestamp for random part in put request', default=str(int(time.time())))
            count = sdk2.parameters.Integer('Count of buckets uniq ammo to generate', default=60000)
            profile = sdk2.parameters.JSON('Shooting profile', default=DefaultProfile, required=True)
            bucket = sdk2.parameters.Integer('Number of buckets for put crequests', default=1)

        with sdk2.parameters.Group('Authorization ammo parameters') as authammo_block:
            keyIDPtr = sdk2.parameters.String('Vault record name with public key', default='key-s3-auth-token')
            secretPtr = sdk2.parameters.String('Vault record name with private key', default='secret-s3-auth-token')
            hostPtr = sdk2.parameters.String('WMI host', default='storage03sas.svc.cloud-preprod.yandex.net:80')
            with sdk2.parameters.RadioGroup('Use autherization?') as auth:
                auth.values['yes'] = auth.Value(value='yes')
                auth.values['no'] = auth.Value(value='no', default=True)

        with sdk2.parameters.Group('Uploaded files') as sources_block:
            loadfile = sdk2.parameters.String('Load config file', default='https://proxy.sandbox.yandex-team.ru/1387531185')
            fileurl = sdk2.parameters.String('Files with keys', default='https://proxy.sandbox.yandex-team.ru/1387198047')
            ammogen = sdk2.parameters.String('Ammo generator binary', default='https://proxy.sandbox.yandex-team.ru/1398588693')
            pandora = sdk2.parameters.String('Pandora binary', default='https://proxy.sandbox.yandex-team.ru/1385033337')

        with sdk2.parameters.Group('The choice of the tank') as tank_coice_block:
            with sdk2.parameters.RadioGroup('Use tank', required=True) as use_tank:
                # use_tank.values['iapet-1.iapet.mail-tanks.tanks.mail.stable.qloud-d.yandex.net'] = use_tank.Value(value='IAPET', default=True)
                use_tank.values['tank01h.load.mdst.yandex.net'] = use_tank.Value(value='Tank01h', default=True)
                use_tank.values['tank-s3.svc.cloud-preprod.yandex.net'] = use_tank.Value(value='TankS3')

        with sdk2.parameters.Output:
            lunapark_job_id = sdk2.parameters.String('Lunapark job id', default_value='')
            lunapark_link = sdk2.parameters.String('Lunapark link', default_value='')

    def start_shooting(self, desc, config_content):
        tanks = []
        tanks.append(self.Parameters.use_tank)
        container = YANDEX_TANKAPI_LXC_CONTAINER.find(state=ctr.State.READY).order(-YANDEX_TANKAPI_LXC_CONTAINER.id).first().id
        if container is not None:
            self.Context.lxc_container = str(container)
        shooting_task = ShootViaTankapi(
            self,
            description=desc,
            ammo_source='in_config',
            config_source='file',
            config_content=config_content,
            tanks=tanks,
            container=container
            ).enqueue()
        self.loger.info('Subtask with shooting is started')
        self.Context.shooting_task_id = str(shooting_task.id)
        raise sdk2.WaitTask([shooting_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)

    def find_shooting(self):
        shooting_find = sdk2.Task.find(
            id=self.Context.shooting_task_id,
            children=True
        ).first()
        self.Context.shooting_find_id = str(shooting_find.id)
        self.loger.info('found ammo_generator_task_id = %s', shooting_find.id)
        self.Parameters.lunapark_job_id = shooting_find.Parameters.lunapark_job_id
        self.Parameters.lunapark_link = shooting_find.Parameters.lunapark_link

    def make_ammo(self):
        ammo = AMMO_FILE(self, 's3 ammo', 'ammo.json', ttl=1)
        ammoData = sdk2.ResourceData(ammo)
        try:
            ammoData.path.write_bytes(self.get_ammo_data())
            ammoData.ready()
            self.Context.ammo_resource = str(ammoData.path)
            return ammo
        except Exception as ex:
            raise ce.TaskFailure("Can not create ammo resource. Exception: {}".format(ex))

    def get_ammo_data(self):
        self.loger.info('Start ammo generation')
        get_source(self.Parameters.ammogen, 'ammogen')
        acommand = "chmod +x ./ammogen && ./ammogen -t '{}' -c {:d} -j '{}' -b {:d} -f '{}' -k '{}' -s '{}' -h '{}' -a '{}'".format(
            self.Parameters.ts,                             # Timestamp
            self.Parameters.count,                          # Count
            json.dumps(self.Parameters.profile),            # Profile
            self.Parameters.bucket,                         # Bucket
            self.Parameters.fileurl,                        # File URL
            sdk2.Vault.data(self.Parameters.keyIDPtr),      # KeyPtr
            sdk2.Vault.data(self.Parameters.secretPtr),     # SecretPtr
            self.Parameters.host,                           # WMI Host
            self.Parameters.auth                            # Auth
        )
        # self.Context.ammo_command = acommand
        aprocess = subprocess.Popen(acommand, bufsize=0, preexec_fn=os.setsid, close_fds=True, shell=True, stdout=subprocess.PIPE)
        data = aprocess.stdout.read()
        return data

    def make_conf(self):
        autostop = []
        scheduler = []
        config_content = fu.read_file(get_source(self.Parameters.loadfile, 'load.yaml'))
        yaml_config = yaml.safe_load(config_content)
        # Modify config for current task
        yaml_config['pandora']['pandora_cmd'] = self.Parameters.pandora.encode('utf-8')
        yaml_config['pandora']['config_content']['pools'][0]['id'] = "S3 pool"
        yaml_config['pandora']['config_content']['pools'][0]['gun']['type'] = "s3-gun"
        yaml_config['pandora']['config_content']['pools'][0]['gun']['target'] = self.Parameters.host.encode('utf-8')
        yaml_config['pandora']['config_content']['pools'][0]['ammo']['type'] = "s3-provider"
        yaml_config['pandora']['config_content']['pools'][0]['startup']['times'] = self.Parameters.instances
        # Ammo resource
        yaml_config['pandora']['resources'][0]['src'] = self.Context.ammo_url
        yaml_config['pandora']['resources'][0]['dst'] = './ammo.json'
        # Uploader section
        yaml_config['uploader']['api-address'] = "https://lunapark.yandex-team.ru/"
        yaml_config['uploader']['job_name'] = "[s3][{}][{}]".format(self.Parameters.scheduler, self.Parameters.comment).encode('utf-8')
        yaml_config['uploader']['operator'] = "lunapark"
        yaml_config['uploader']['task'] = self.Parameters.task.encode('utf-8')

        for condition in self.Parameters.autostop:
            autostop.append(condition.encode('utf-8'))
        yaml_config['autostop']['autostop'] = autostop
        for rps in self.Parameters.scheduler:
            scheduler.append(ast.literal_eval(rps.encode('utf-8')))
        yaml_config['pandora']['config_content']['pools'][0]['rps'] = scheduler
        self.loger.info("scheduler is {}".format(yaml_config['pandora']['config_content']['pools'][0]['rps']))

        return yaml.dump(yaml_config, default_flow_style=False, encoding=None)

    def on_execute(self):
        self.loger = logger()
        self.loger.info("Start S3 shooting")

        desc = self.Parameters.comment

        with self.memoize_stage.make_ammo:
            ammo_resource = self.make_ammo()
            self.Context.ammo_url = str(ammo_resource.http_proxy)

        with self.memoize_stage.make_conf:
            config_content = self.make_conf()

        with self.memoize_stage.shooting:
            self.start_shooting(desc, config_content)

        if self.Context.shooting_task_id != "":
            self.find_shooting()

        self.loger.info("Finish S3 shooting")
# ==== End Of Class ==== #


def logger():
    loggerr = logging.getLogger('%s_%s' % (__name__, time.time()))
    loggerr.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        's3_shooting.log',
        maxBytes=(1024 * 1024),
        backupCount=5
    )

    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    loggerr.addHandler(file_handler)
    return loggerr


def get_source(url, dst):
    session = requests.session()
    try:
        with open(dst, 'wb') as resource:
            resource.write(session.get(url, stream=True).content)
        return os.path.abspath(dst)
    except Exception as ex:
        raise ce.TaskFailure("Can't download resource. {}".format(ex))
    finally:
        session.close()
