"""Import modules"""
import logging
import requests
import json
from sandbox import sdk2


class ClickHouseDropDetachedParts(sdk2.Task):
    """Drop Detached Parts from ClickHouse"""

    class Parameters(sdk2.Parameters):
        """Parameters"""
        hosts = sdk2.parameters.List("vla-xxxxxxxxxxxxxxxx.db.yandex.net")
        clickhouse_secret = sdk2.parameters.String("secret db_name, user and pass", do_not_copy=True)
        tag_notify = sdk2.parameters.String("Tag notify")

    def select_detached_parts(self, host):
        """Query select detached parts from CH"""
        query = 'SELECT name FROM system.detached_parts'
        url = f'https://{host}:8443/?database={self.db}&query={query}'
        response = requests.get(url, headers=self.headers, verify=self.ca_cert)
        if response.text is None:
            return []
        else:
            return response.text.split("\n")

    def alter_detached_part(self, host, query):
        """Query Alter table drop detached part by name"""
        url = f'https://{host}:8443/?database={self.db}&query={query}'
        response = requests.post(url, headers=self.headers, verify=self.ca_cert)
        return response.text

    def main(self):
        """Main"""
        secret = sdk2.yav.Secret(self.Parameters.clickhouse_secret)
        self.db = secret.data()["db_name"]
        self.headers = {
            "X-ClickHouse-User": secret.data()["user"],
            "X-ClickHouse-Key": secret.data()["pass"]
        }
        self.ca_cert = '/usr/share/yandex-internal-root-ca/YandexInternalRootCA.crt'
        for host in self.Parameters.hosts:
            for name_part in self.select_detached_parts(host):
                query = f"""ALTER TABLE `kinopoisk-balancer-logs`.access_log
                        DROP DETACHED PART '{name_part}' SETTINGS allow_drop_detached = 1"""
                self.alter_detached_part(host, query)
        return True

    def juggler_send(self, status, desc):
        """Send event to juggler api"""
        juggler_events = {"events": []}
        juggler_events["events"].append({"host": "ClickHouse_Balancer_Logs",
                                         "description": desc,
                                         "service": "Drop_Detached_Parts", "status": status,
                                         "tags": [self.Parameters.tag_notify]})
        headers = {
            "Content-Type": "application/json"
        }
        url = "https://juggler-push.search.yandex.net/events"
        resp = requests.post(url, data=json.dumps(juggler_events), headers=headers)
        logging.info(resp.text)

    def on_execute(self):
        try:
            self.main()
            self.juggler_send("OK", "OK")
        except RuntimeError as error:
            logging.error(f"ERROR: {error}\n")
            self.juggler_send("CRIT", "Error script:" + error)
