"""Import modules"""
import logging
import json
import requests
import os
from sandbox import sdk2
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.process import run_process


class MediaQuotasMonitoring(sdk2.Task):
    """Monitoring Quotas YD"""

    class Parameters(sdk2.Parameters):
        """Parameters"""
        services = sdk2.parameters.List('Ex.: {"id": 645, "name": "kinopoisk"}')
        secret_path = sdk2.parameters.String("secret yp_token and ya-token", do_not_copy=True)
        tag_notify = sdk2.parameters.String("Tag notify")

    juggler_events = {"events": []}
    # List of DataCenters
    dcs = ["SAS", "IVA", "MAN", "VLA", "MYT"]
    # Args for monitoring
    args = ["vcpu", "memory", "ssd", "ssd_bw", "hdd", "hdd_bw", "net_bw"]
    # link to quotes
    link = "https://deploy.yandex-team.ru/quotas"
    path_yp_util = None

    def get_path_yp_util(self):
        """Start ya tool and getting path for yp-util and environment"""
        ya_tool = svn.Arcadia.export('arcadia:/arc/trunk/arcadia/ya', os.path.realpath('ya'))
        path_yp_util = f"{ya_tool} tool yp-util"
        logging.info(f"path_yo_util: {path_yp_util}")
        secret = sdk2.yav.Secret(self.Parameters.secret_path)
        yp_token = secret.data()["yp-token"]
        ya_token = secret.data()["ya_token"]
        os.environ["YA_TOKEN"] = ya_token
        os.environ["YP_TOKEN"] = yp_token

        # First Run_process yp-util for download without param
        run_process(path_yp_util.split(), log_prefix='yp-util_out_download').wait()
        return path_yp_util

    def get_quotas(self, cluster, service_id):
        """Getting usage quotas by cluster and service id"""
        cmd = f"{self.path_yp_util} account explain abc:service:{service_id} --cluster {cluster}"\
                f" --format json --no-abc-resolving --no-abc-resolving"
        cmd = cmd.split()
        run_process(cmd, log_prefix=f"yp-util_out_{service_id}_{cluster}").wait()
        with open(f"log1/yp-util_out_{service_id}_{cluster}.out.txt") as out:
            try:
                return {"status": True, "data": json.loads(out.read())}
            except ValueError as error_text:
                return {"status": False, "error": error_text}

    def calc_percent(self, quotes, arg):
        """Calculation of the percentage of use"""
        if quotes["acc_limits"][arg] != 0:
            return quotes["acc_usages"][arg] / quotes["acc_limits"][arg] * 100
        return 0

    def get_acc_by_seg(self, quota_acc, segment):
        """Getting accounts limits or usages by segment"""
        for accounts in quota_acc:
            if accounts["segment"] == segment:
                return accounts
        return {}

    def converting_format(self, arg, value):
        """Converting format unit"""
        if arg in ("memory", "ssd", "ssd_bw", "hdd", "hdd_bw", "net_bw"):
            if value > 1024 ** 4:
                unit = "TiB"
                new_value = value / 1024 ** 4
            elif value > 1024 ** 3:
                unit = "GiB"
                new_value = value / 1024 ** 3
            elif value > 1024 ** 2:
                unit = "MiB"
                new_value = value / 1024 ** 2
            elif value > 1024:
                unit = "KiB"
                new_value = value / 1024
            else:
                unit = "Bytes"
                new_value = value
            if "_bw" in arg:
                unit += "/s"
        elif arg == "vcpu":
            unit = "vcores"
            new_value = value / 1000
        return {"value": round(new_value, 2), "format": unit}

    def juggler_events_add(self, event):
        """Add event to var juggler_events"""
        self.juggler_events["events"].append(event)

    def juggler_push(self):
        """Send events to juggler api"""
        headers = {
            "Content-Type": "application/json"
        }
        url = "https://juggler-push.search.yandex.net/events"
        resp = requests.post(url, data=json.dumps(self.juggler_events), headers=headers)
        logging.info(resp.text)

    def main(self):
        """Main"""
        for service in self.Parameters.services:
            if self.path_yp_util is None:
                self.path_yp_util = self.get_path_yp_util()
            service = json.loads(service)
            desc = f"YP Quotas: Service {service['name']}:"
            alert_status = False
            tags = [service["name"], "YP", "Quotes", self.Parameters.tag_notify]
            for data_center in self.dcs:
                desc_tmp = f"\n[{data_center}]:"
                alert_status_dc = False
                quotas = {}
                quota_tmp = self.get_quotas(data_center, service["id"])
                if quota_tmp["status"]:
                    if not quota_tmp["data"]["accounts_usages"]:
                        continue
                    quotas["acc_usages"] = self.get_acc_by_seg(quota_tmp["data"]["accounts_usages"],
                                                                   "default")
                    quotas["acc_limits"] = self.get_acc_by_seg(quota_tmp["data"]["accounts_limits"],
                                                                   "default")

                    for arg in self.args:
                        quote_perc = round(self.calc_percent(quotas, arg), 2)
                        if quote_perc >= 90:
                            alert_status_dc = True
                            use = self.converting_format(arg, quotas["acc_usages"][arg])
                            limit = self.converting_format(arg, quotas["acc_limits"][arg])
                            desc_tmp += f" {arg}: [{use['value']}{use['format']}|{limit['value']}"\
                                        f"{limit['format']}] ({quote_perc}%)"
                    if alert_status_dc:
                        desc += desc_tmp
                        alert_status = True
                    continue

                self.juggler_events_add({"host": service["name"] + "_YP",
                                         "description": "Error Script: " + quota_tmp["error"],
                                         "service": "Quotas", "status": "WARN", "tags": tags})
                self.juggler_push()
                return False
            if alert_status:
                status = "CRIT"
            else:
                desc += "All OK"
                status = "OK"
            desc += f"\nLink: {self.link}"
            self.juggler_events["events"].append({"host": service["name"] + "_YP",
                                                  "description": desc, "service": "Quotas",
                                                  "status": status, "tags": tags})

        self.juggler_push()
        return True

    def on_execute(self):
        try:
            self.main()
        except RuntimeError as error:
            logging.error(f"ERROR: {error}\n")
