# -*- coding: utf-8 -*-
"""Visualize backups steps"""
import argparse
import logging as log
from pprint import pprint
from datetime import timedelta, datetime
from sandbox.projects.media.admins.MediaRotateS3Backups import BackupItem, BackupRotate, Policy  # , S3Client

TPL = "backup/mysql/name/%F"


def main():
    """Helper for view rotate process"""

    parser = argparse.ArgumentParser(description='Visualize "rotate-s3-backups"')
    parser.add_argument('--start', metavar='YYYMMDD', help="Start from time (default 20170903)", default="20170903")
    parser.add_argument('--che', metavar='YYYYMMDD', nargs="+", help="Time check points", required=True)
    parser.add_argument('--debug', help="Debug logs", action='store_true', required=False, default=False)
    parser.add_argument('-p', '--period', help="Backup period in days (def=365 days)", default=365, type=int)
    parser.add_argument('-d', '--daily', help="Policy daily (def=5)", default=5, type=int)
    parser.add_argument('-w', '--weekly', help="Policy weekly (def=1)", default=1, type=int)
    parser.add_argument('-m', '--monthly', help="Policy monthly (def=1)", default=1, type=int)
    args = parser.parse_args()

    log.basicConfig(
        format="%(asctime)s %(levelname)5s %(funcName)s:%(lineno)-4s %(message)s",
        level=log.DEBUG if args.debug else log.ERROR,
    )

    time_start = datetime.strptime(args.start, "%Y%m%d")
    che_times = []
    for che_time in args.che:
        che_times.append(datetime.strptime(che_time, "%Y%m%d"))
    print(f"Start time {time_start.strftime('%F')}")
    print(f"Time check points: {[x.strftime('%F') for x in che_times]}")

    policy = Policy(
        daily=args.daily,
        weekly=args.weekly,
        monthly=args.monthly
    )
    worker = BackupRotate(from_last=True, policy=policy, s3client=None)

    backups = []
    for itr in range(args.period):
        current_date = BackupItem(path=(time_start + timedelta(days=itr)).strftime(TPL))
        backups.append(current_date)

        rotation_ts = worker.get_now_timestamp(backups)
        backup_groups = worker.group_by_frequency(backups)
        worker.select_valid(backup_groups, rotation_ts, policy)

        preserve_backups = []
        for bkp in backups:
            if bkp.state == BackupItem.NONE:
                continue

            # reset backup state
            bkp.state = BackupItem.NONE
            preserve_backups.append(bkp)

        # save only valid backups
        backups = preserve_backups
        if current_date.timestamp in che_times:
            print(f"Rotate from {rotation_ts}")
            print(f"Backups for {current_date.timestamp.strftime('%F')}")
            pprint([x.timestamp.strftime("%F") for x in preserve_backups])


if __name__ == "__main__":
    main()
