import logging
import os
import json
import requests
import sys
from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.process import run_process, SandboxSubprocessError


class InvalidSecretData(Exception):
    pass


class MysqlTablesToYt(sdk2.Task):
    """ Export tables from MySQL to YT """

    class Requirements(sdk2.Requirements):
        cores = 4
        ram = 2048
        client_tags = ctc.Tag.LINUX_FOCAL
        container_resource = 3078372951

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        mysql_host = sdk2.parameters.String('MySQL host', required=True)
        mysql_port = sdk2.parameters.Integer('MySQL port', default=3306, required=True)
        mysql_user = sdk2.parameters.String('MySQL user', required=True)
        yav_mysql_password = sdk2.parameters.YavSecret('YAV secret with MySQL password', required=True)
        yav_yt_token = sdk2.parameters.YavSecret('YAV secret with YT token', required=True)
        config = sdk2.parameters.JSON('Jobs config', required=True, default={
            "templates": {
                "template1": {
                    "retry": 3,
                    "yt_cluster": "cluster1",
                    "yt_path": "//path_to_dir1",
                    "database": "db1"
                }
            },
            "jobs": {
                "job1": {
                    "tables": [
                        "table1",
                        "table2"
                    ],
                    "template": "template1"
                },
                "job2": {
                    "yt_table": "yt_table_name",
                    "yt_cluster": "cluster2",
                    "yt_path": "//path_to_dir2",
                    "columns": [
                        "column1",
                        "column2",
                        "column3"
                    ],
                    "tables": [
                        "table1"
                    ],
                    "database": "db2"
                }
            }
        })
        juggler_host = sdk2.parameters.String('Juggler host', default='juggler.sb.mysql_tables_to_yt', required=True)
        juggler_service = sdk2.parameters.String('Juggler service', required=True)
        juggler_tags = sdk2.parameters.List('Juggler tags', required=False)

    def on_execute(self):
        def http_response(response, decode_json=False):
            """ Get response content """
            if response.status_code != requests.codes.ok:
                sys.exit('Error {} [{}], {}: {}'.format(
                    response.status_code,
                    response.url,
                    response.reason,
                    response.text)
                )
            content = response.json() if decode_json else response.text
            return content

        def request_post(url, headers, post_content, decode_json=False):
            """ Make POST request """
            response = requests.post(url, headers=headers, json=post_content, timeout=10)
            content = http_response(response, decode_json)
            return content

        def send_event_to_juggler(juggler_host, juggler_service, juggler_status, juggler_description, juggler_tags):
            """ Send event to Juggler """
            url = 'http://juggler-push.search.yandex.net/events'
            events = {
                'host': juggler_host,
                'service': juggler_service,
                'status': juggler_status,
                'description': juggler_description
            }
            if juggler_tags:
                events['tags'] = juggler_tags
            post_content = {
                'source': 'sandbox',
                'events': [
                    events
                ]
            }
            headers = {
                'Content-Type': 'application/json'
            }
            content = request_post(url, headers, post_content, True)
            logging.info('Successfully sent event to Juggler'
                         if content['events'][0]['code'] == 200 else 'Failed to send event to Juggler')
            return content

        # Fill variables from SB
        mysql_host = self.Parameters.mysql_host
        mysql_port = self.Parameters.mysql_port
        mysql_user = self.Parameters.mysql_user
        yav_mysql_password = self.Parameters.yav_mysql_password
        yav_yt_token = self.Parameters.yav_yt_token
        config = self.Parameters.config
        juggler_host = self.Parameters.juggler_host
        juggler_service = self.Parameters.juggler_service
        juggler_tags = self.Parameters.juggler_tags

        logging.info('MySQL host: {}'.format(mysql_host))
        logging.info('MySQL port: {}'.format(mysql_port))
        logging.info('MySQL user: {}'.format(mysql_user))
        logging.info('YAV secret with MySQL password: {}'.format(yav_mysql_password))
        logging.info('YAV secret with YT token: {}'.format(yav_yt_token))
        logging.info('Jobs config: {}'.format(config))
        logging.info('Juggler host: {}'.format(juggler_host))
        logging.info('Juggler service: {}'.format(juggler_service))
        logging.info('Juggler tags: {}'.format(juggler_tags))

        # Get secrets from YAV
        logging.info('Getting YAV secrets')
        mysql_password = yav_mysql_password.value()
        yt_token = yav_yt_token.value()

        # Make script file
        logging.info('Writing script file')
        script = r"""#!/usr/bin/env perl

use strict;
use warnings;

use JSON::PP;
use POSIX qw(strftime);

no if $] >= 5.018, warnings => qw(experimental::smartmatch);

use IPC::Run qw(run);

my $config;
while (<>) {$config .= $_;}
$config = decode_json($config);

my $global_error = 0;

sub current_date {
    return strftime("%Y.%m.%d %H:%M:%S", localtime);
}

sub check_error {
    my ($error, $msg, $path) = @_;
    my $ret_code = 0;
    if ($error) {
        printf "%s: %s\n\n", current_date(), $msg;
        $global_error = 1;
        if (-f $path) {unlink $path;}
        $ret_code = 1;
    }
    return $ret_code;
}

foreach my $job (keys %{$config->{"jobs"}}) {
    my %job_config;
    if (exists($config->{"jobs"}->{$job}->{"template"})) {
        my $template = $config->{"jobs"}->{$job}->{"template"};
        if (exists($config->{"templates"}->{$template})) {
            %job_config = %{$config->{"templates"}->{$template}};
        }
    }
    foreach my $param (keys %{$config->{"jobs"}->{$job}}) {
        $job_config{$param} = $config->{"jobs"}->{$job}->{$param};
    }

    my $retry = exists($job_config{"retry"}) ? $job_config{"retry"} : 1;

    foreach (qw(tables database yt_path yt_cluster)) {
        if (not exists($job_config{$_})) {
            printf "Job '%s' missing parameter '%s'\n", $job, $_;
            exit 1;
        }
    }

    printf "%s: job '%s', DB '%s', YT cluster '%s'\n",
        current_date(), $job, $job_config{"database"}, $job_config{"yt_cluster"};

    foreach my $table (@{$job_config{"tables"}}) {
        printf "\ttable '%s'\n", $table;
        my @columns;
        my $mysql_dump_path = sprintf("/tmp/mysql_%s.sql_dump.zst", $table);
        my $error = 0;
        if (exists($job_config{"columns"})) {@columns = @{$job_config{"columns"}};}
        else {
            my ($cmd_out, $cmd_err);
            my @cmd = (
                '/usr/bin/mysql',
                '-NB',
                $job_config{"database"},
                '-e',
                sprintf(
                    "SELECT `COLUMN_NAME` FROM INFORMATION_SCHEMA.COLUMNS WHERE `TABLE_SCHEMA` = \"%s\" AND `TABLE_NAME` = \"%s\"",
                    $job_config{"database"}, $table
                )
            );
            printf "%s: get columns names from MySQL\n",
                current_date();
            my $status = run \@cmd, '>', \$cmd_out, '2>', \$cmd_err;
            if (defined($cmd_err) and ($cmd_err ne '')) {
                chomp $cmd_err;
                printf "\tCommand error: [%s]\n", $cmd_err;
            }
            if ($status) {
                $error = 0;
                foreach (split("\n", $cmd_out)) {
                    chomp;
                    push @columns, $_;
                }
            }
            else {$error = 1;}
            if (check_error(
                $error, "can't get columns names from MySQL",
                $mysql_dump_path
            )
            ) {
                next;
            }
        }
        my @sort_columns;
        if (exists($job_config{"sort_columns"})) {
            @sort_columns = @{$job_config{"sort_columns"}};
            my @columns_ordered = @sort_columns;
            foreach (@columns) {
                if ($_ ~~ @sort_columns) {next;}
                else {push @columns_ordered, $_;}
            }
            @columns = @columns_ordered;
        }
        my @columns_for_select;
        my @ifnull_columns = exists($job_config{"ifnull_columns"}) ? @{$job_config{"ifnull_columns"}} : ();
        my $ifnull_replace = exists($job_config{"ifnull_replace"}) ? $job_config{"ifnull_replace"} : "";
        foreach (@columns) {
            push(@columns_for_select,
                $_ ~~ @ifnull_columns
                    ? sprintf("IFNULL(`%s`, '%s') AS `%s`", $_, $ifnull_replace, $_)
                    : "`$_`"
            );
        }

        $error = 0;
        printf "\tcolumns '%s'\n", join(", ", @columns);
        if (@sort_columns) {printf "\tsort columns '%s'\n", join(", ", @sort_columns);}
        if (@ifnull_columns) {printf "\tIFNULL columns '%s'\n", join(", ", @ifnull_columns);}
        for (my $i = 1; $i <= $retry; $i++) {
            my $cmd_err;
            my @cmd = (
                '/usr/bin/mysql',
                '-NB',
                '--quick',
                $job_config{"database"},
                '-e',
                sprintf(
                    "SET NAMES utf8; SELECT %s FROM `%s`%s",
                    join(", ", @columns_for_select),
                    $table,
                    @sort_columns
                        ? sprintf(" ORDER BY %s",
                        join(", ", map {"`$_`"} @sort_columns))
                        : ""
                )
            );
            my $compress_err;
            my @compress_cmd = qw(/usr/bin/zstd -T0 -cq);
            printf "%s: export from MySQL [try %d/%d]\n",
                current_date(), $i, $retry;
            my $status = run \@cmd, '2>', \$cmd_err, '|', \@compress_cmd, '>', $mysql_dump_path, '2>', \$compress_err;
            if (defined($cmd_err) and ($cmd_err ne '')) {
                chomp $cmd_err;
                printf "\tCommand error: [%s]\n", $cmd_err;
            }
            if (defined($compress_err) and ($compress_err ne '')) {
                chomp $compress_err;
                printf "\tCompress command error: [%s]\n", $compress_err;
            }
            printf "%s: export from MySQL %s\n",
                current_date(),
                $status ? "done" : "failed";
            if ($status) {
                $error = 0;
                last;
            }
            else {$error = 1;}
        }
        if (check_error(
            $error, "can't export from MySQL", $mysql_dump_path
        )
        ) {
            next;
        }

        $error = 0;
        my $yt_full_path = sprintf("%s/%s",
            $job_config{"yt_path"},
            exists($job_config{"yt_table"})
                ? $job_config{"yt_table"}
                : $table);
        my $yt_full_path_tmp = $yt_full_path . "_tmp";
        printf "\tYT tmp path '%s'\n", $yt_full_path_tmp;
        my $schema;
        if (@sort_columns) {
            my @schema_arr;
            foreach (@columns) {
                if ($_ ~~ @sort_columns) {
                    push @schema_arr,
                        sprintf(
                            "{name = %s; type = int64; sort_order = \"ascending\"}",
                            $_);
                }
                else {
                    push @schema_arr,
                        sprintf("{name = %s; type = utf8}", $_);
                }
            }
            $schema = sprintf("<schema=[%s]>%s",
                join("; ", @schema_arr),
                $yt_full_path_tmp);
        }
        for (my $i = 1; $i <= $retry; $i++) {
            my ($cmd_out, $cmd_err);
            my @cmd = ('/usr/bin/yt', 'write');
            push @cmd, defined($schema) ? $schema : $yt_full_path_tmp;
            push @cmd,
                (
                    '--format',
                    sprintf(
                        "<columns=[%s];enable_type_conversion=%%true>schemaful_dsv",
                        join("; ", @columns)),
                    '--proxy',
                    $job_config{"yt_cluster"}
                );
            my $decompress_err;
            my @decompress_cmd = qw(/usr/bin/zstd -T0 -cdq);
            printf "%s: upload to YT [try %d/%d]\n",
                current_date(), $i, $retry;
            my $status = run(
                \@decompress_cmd, '<', $mysql_dump_path, '2>', \$decompress_err,
                '|',
                \@cmd, '>', \$cmd_out, '2>', \$cmd_err
            );
            if (defined($cmd_out) and ($cmd_out ne '')) {
                chomp $cmd_out;
                printf "\tCommand output: [%s]\n", $cmd_out;
            }
            if (defined($cmd_err) and ($cmd_err ne '')) {
                chomp $cmd_err;
                printf "\tCommand error: [%s]\n", $cmd_err;
            }
            if (defined($decompress_err) and ($decompress_err ne '')) {
                chomp $decompress_err;
                printf "\tDecompress command error: [%s]\n", $decompress_err;
            }
            printf "%s: upload to YT %s\n",
                current_date(),
                $status ? "done" : "failed";
            if ($status) {
                $error = 0;
                last;
            }
            else {$error = 1;}
        }
        if (check_error($error, "can't upload to YT", $mysql_dump_path)) {
            next;
        }

        {
            my ($cmd_out, $cmd_err);
            my @cmd = (
                '/usr/bin/yt', 'exists', $yt_full_path, '--proxy',
                $job_config{"yt_cluster"}
            );
            my $status = run \@cmd, '>', \$cmd_out, '2>', \$cmd_err;
            chomp $cmd_out;
            if ($status and ($cmd_out eq "true")) {
                @cmd = (
                    '/usr/bin/yt', 'remove', $yt_full_path, '--proxy',
                    $job_config{"yt_cluster"}
                );
                run \@cmd, '>', \$cmd_out, '2>', \$cmd_err;
            }
        }

        printf "\tYT path '%s'\n", $yt_full_path;
        {
            my ($cmd_out, $cmd_err);
            my @cmd = (
                '/usr/bin/yt', 'move', $yt_full_path_tmp, $yt_full_path,
                '--proxy', $job_config{"yt_cluster"}
            );
            printf "%s: move table on YT\n",
                current_date();
            my $status = run \@cmd, '>', \$cmd_out, '2>', \$cmd_err;
            if (defined($cmd_out) and ($cmd_out ne '')) {
                chomp $cmd_out;
                printf "\tCommand output: [%s]\n", $cmd_out;
            }
            if (defined($cmd_err) and ($cmd_err ne '')) {
                chomp $cmd_err;
                printf "\tCommand error: [%s]\n", $cmd_err;
            }
            printf "%s: move table on YT %s\n",
                current_date(),
                $status ? "done" : "failed";
            if ($status) {$error = 0;}
            else {$error = 1;}
            if (check_error(
                $error, "can't move table on YT",
                $mysql_dump_path
            )
            ) {
                next;
            }
        }

        print "\n";

        if (-f $mysql_dump_path) {unlink $mysql_dump_path;}
    }
}

if ($global_error) {exit 1;}
"""
        script_path = os.environ['HOME'] + '/mysql-tables-to-yt.pl'
        with open(script_path, 'w') as fh:
            fh.write(script)

        # Make config files
        logging.info('Writing config files')

        conf_path = os.environ['HOME'] + '/mysql-tables-to-yt.json'
        with open(conf_path, 'w') as fh:
            json.dump(config, fh, indent=2)

        mysql_conf = os.environ['HOME'] + '/.my.cnf'
        with open(mysql_conf, 'w') as fh:
            fh.write('[client]\n')
            fh.write('host={}\n'.format(mysql_host))
            fh.write('port={}\n'.format(mysql_port))
            fh.write('user={}\n'.format(mysql_user))
            fh.write('password={}\n'.format(mysql_password))

        # Export environment variables
        logging.info('Exporting environment variables')
        os.environ['YT_TOKEN'] = yt_token

        # Run mysql-tables-to-yt
        logging.info('Running mysql-tables-to-yt')
        cmd = ['/usr/bin/perl', script_path, conf_path]
        logging.info('Command: {}'.format(cmd))
        juggler_status = 'OK'
        juggler_description = 'OK'
        try:
            run_process(cmd)
        except SandboxSubprocessError as err:
            logging.info('Error: {}'.format(err))
            juggler_status = 'CRIT'
            juggler_description = 'See https://sandbox.yandex-team.ru/task/{}/view'.format(sdk2.Task.current.id)

        # Send to Juggler
        send_event_to_juggler(juggler_host, juggler_service, juggler_status, juggler_description, juggler_tags)
