import logging
import requests
import sandbox.common.errors as sb_errors
import sandbox.projects.common.decorators as sb_decorators
from sandbox import sdk2
from datetime import datetime
from dateutil import parser

STATUSES = ["OK", "WARN", "CRIT"]


class ProgressCounter(object):

    def __init__(self, total):
        self.total = total
        self.curr = 0

    @property
    def progress(self):
        return "{}/{}".format(self.curr, self.total)

    def count(self):
        self.curr += 1


class PlatformClient(object):
    PLATFORM_API = "https://platform.yandex-team.ru/api/v1"

    def __init__(self, token):
        self.token = token

    def _request(self, endpoint, method="get", timeout=60, **kwargs):
        url = "{}{}".format(self.PLATFORM_API, endpoint)
        headers = {
            "Authorization": "OAuth {}".format(self.token)
        }
        headers.update(kwargs.pop("headers", {}))
        return getattr(requests, method)(url, headers=headers, timeout=timeout, **kwargs)

    @sb_decorators.retries(max_tries=3, delay=10)
    def project(self, project):
        endpoint = "/project/{}".format(project)
        result = self._request(endpoint)
        result.raise_for_status()
        return result

    @sb_decorators.retries(max_tries=3, delay=10)
    def domains(self, object_id):
        endpoint = "/domains/{}".format(object_id)
        result = self._request(endpoint)
        result.raise_for_status()
        return result


class PlatformCheckCerts(sdk2.Task):
    """ Check certificates expiration in Platform """

    BINARY_TASK_ATTR_TARGET = "media/admins/PlatformCheckCerts"

    class Requirements(sdk2.Task.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):

        yav_qloud_token_secret = sdk2.parameters.YavSecret("YAV secret with Qloud token", required=True)
        yav_qloud_token_secret_key = sdk2.parameters.String("YAV key name of secret", default="token", required=True)
        projects = sdk2.parameters.List("Platform projects", required=True)
        warn_diff = sdk2.parameters.Integer("WARN diff", default=30, required=True)
        crit_diff = sdk2.parameters.Integer("CRIT diff", default=14, required=True)
        j_host = sdk2.parameters.String("Juggler host", default="juggler.sandbox.platform_check_certs", required=True)
        j_service = sdk2.parameters.String("Juggler service", required=True)
        j_tags = sdk2.parameters.List("Juggler tags")
        print_task_url = sdk2.parameters.Bool("Print task URL", required=True, default=False)

        with sdk2.parameters.Group("Task resource settings", collapse=True) as build_settings_block:
            UseLastBinary = sdk2.parameters.Bool("Use last binary archive", default=True)
            with UseLastBinary.value[True]:
                with sdk2.parameters.RadioGroup("Binary release type") as ReleaseType:
                    ReleaseType.values.stable = ReleaseType.Value("stable", default=True)
                    ReleaseType.values.test = ReleaseType.Value("test")
            with UseLastBinary.value[False]:
                custom_tasks_archive_resource = sdk2.parameters.Resource("task archive resource", default=None)

    def on_save(self):
        if self.Parameters.UseLastBinary:
            self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
                attrs={"target": self.BINARY_TASK_ATTR_TARGET,
                       "release": self.Parameters.ReleaseType or "stable"}
            ).first().id
        else:
            self.Requirements.tasks_resource = self.Parameters.custom_tasks_archive_resource

    @property
    def projects(self):
        return self.Parameters.projects

    @property
    def p_t_secret(self):
        return self.Parameters.yav_qloud_token_secret

    @property
    def p_t_secret_key(self):
        return self.Parameters.yav_qloud_token_secret_key

    @property
    def p_token(self):
        return self.p_t_secret.data()[self.p_t_secret_key]

    @property
    def warn_diff(self):
        return self.Parameters.warn_diff

    @property
    def crit_diff(self):
        return self.Parameters.crit_diff

    @property
    def j_host(self):
        return self.Parameters.j_host

    @property
    def j_service(self):
        return self.Parameters.j_service

    @property
    def j_tags(self):
        return self.Parameters.j_tags

    @property
    def print_task_url(self):
        return self.Parameters.print_task_url

    @property
    def result_tmpl(self):
        return {status: [] for status in STATUSES}

    def check_domain(self, domain, object_id):
        """
        Check if domain will expire soon
        :param domain: domain json, see https://wiki.yandex-team.ru/qloud/doc/api/domains/
        :param object_id: object_id, str()
        :return: tuple(status, message). status: oneOf(OK,WARN,CRIT); message: str()
        """
        import pytz
        ca = domain.get("CA")
        if ca != "CertumProductionCA":
            return None, None
        domain_status = domain["domainStatus"]
        if domain_status != "ACTIVE":
            status = "WARN"
            message = "{}: Unknown domain status '{}'".format(object_id, domain_status)
            logging.info("\t{}".format(message))
            return status, message
        sec_task = domain.get("issueCertificateTicket", "?")
        cert_domains = domain["certificateValidFor"]
        cert_valid_till = domain["certificateValidNotAfter"]
        valid_till_tp = parser.parse(cert_valid_till)
        expires_in = valid_till_tp - pytz.utc.localize(datetime.utcnow())
        logging.info("\tCA: %s, Domains: %s, Valid till: %s, Ticket: %s", ca, cert_domains, cert_valid_till, sec_task)
        message = "{}: '{}' expires in {} days - {}, ticket {}".format(
            object_id, cert_domains, expires_in.days, cert_valid_till, sec_task)

        status = "OK"
        if expires_in.days <= self.crit_diff:
            status = "CRIT"
        elif expires_in.days <= self.warn_diff:
            status = "WARN"
        logging.info("\tJuggler: %s: Expires in %s days", status, expires_in.days)
        return status, message

    def check_env(self, env, apps_counter, envs_counter):
        """
        Check environment for expiring domains
        :param env: env, json()
        :param apps_counter: applications counter object, ProgressCounter()
        :param envs_counter: environments counter object, ProgressCounter()
        :return: self.result_tmpl filled with str()
        """
        result = self.result_tmpl
        obj_id = env["objectId"]
        status = env.get("status")
        logging.info("App: %s, env: %s: %s - %s", apps_counter.progress, envs_counter.progress, obj_id, status)
        if status != "DEPLOYED":
            return
        try:
            domains = self.p_c.domains(obj_id).json()
        except requests.exceptions.HTTPError as error:
            raise sb_errors.TaskError("{} returned bad http code: {}, content: {}".format(
                error.request.url, error.response.status_code, error.response.content))
        for domain in domains:
            status, message = self.check_domain(domain, obj_id)
            if status:
                result[status].append(message)
        return result

    def check_project(self, project):
        """
        Check project for expiring domains
        :param project: project name, str()
        :return: self.result_tmpl filled with str()
        """
        result = self.result_tmpl
        try:
            project_data = self.p_c.project(project).json()
        except requests.exceptions.HTTPError as error:
            raise sb_errors.TaskError("{} returned bad http code: {}, content: {}".format(
                error.request.url, error.response.status_code, error.response.content))

        apps = project_data["applications"]
        apps_progress = ProgressCounter(len(apps))
        for app in apps:
            apps_progress.count()
            envs = app["environments"]
            envs_progress = ProgressCounter(len(envs))
            for env in envs:
                envs_progress.count()
                env_result = self.check_env(env, apps_progress, envs_progress)
                if not env_result:
                    continue
                for k, v in env_result.items():
                    result[k].extend(v)
        return result

    @sb_decorators.retries(max_tries=3, delay=10)
    def juggler(self, result):
        j_description = ""

        def format_descr(status, messages):
            msg = "OK: Ok"
            if status != "OK":
                msg = "{}: {}".format(status, "; ".join(messages))
            return msg

        logging.info("Juggler events: %s", result)
        j_status = "OK"
        if len(result["WARN"]) > 0:
            j_status = "WARN"
            j_description = format_descr(j_status, result[j_status])
        if len(result["CRIT"]) > 0:
            j_status = "CRIT"
            j_description = "{} . {}".format(format_descr(j_status, result[j_status]), j_description)
        if self.print_task_url:
            j_description += " https://sandbox.yandex-team.ru/task/{}/view".format(sdk2.Task.current.id)

        logging.info("Juggler host: %s, service: %s, status: %s, description: %s, tags: %s",
                     self.j_host, self.j_service, j_status, j_description, self.j_tags)
        events = {
            "host": self.j_host,
            "service": self.j_service,
            "status": j_status,
            "description": j_description
        }
        if self.j_tags:
            events["tags"] = self.j_tags
        data = {
            "source": "sandbox",
            "events": [
                events
            ]
        }
        ua = requests.post("http://juggler-push.search.yandex.net/events", json=data).json()
        if ua["events"][0]["code"] != 200:
            msg = "Failed to send event to Juggler, response: {}".format(ua)
            logging.error(msg)
            raise sb_errors.TaskError(msg)
        logging.info("Successfully sent event to Juggler")

    def on_execute(self):
        logging.info("YAV secret/key with platform token: %s/%s", self.p_t_secret, self.p_t_secret_key)
        logging.info("Projects: %s", self.projects)
        logging.info("WARN / CRIT days to expire: %s / %s", self.warn_diff, self.crit_diff)
        logging.info("Juggler host: %s, service: %s", self.j_host, self.j_service)

        self.p_c = PlatformClient(self.p_token)
        result = self.result_tmpl
        for project in self.projects:
            logging.info("Checking project: %s", project)
            project_result = self.check_project(project)
            for k, v in project_result.items():
                result[k].extend(v)
        self.juggler(result)
