import collections
import json
import logging
import os
import re
import sys
import tarfile
import time
import datetime
from dateutil import parser
import pytz

import requests
from sandbox import sdk2
from sandbox.sandboxsdk.process import run_process


class YdAwacsArcadiaRemoveUnstables(sdk2.Task):
    """ Remove Arcadia unstables from YD and AWACS """

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 3072

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):

        yav_yp_token = sdk2.parameters.YavSecret('YAV secret with YP token', required=True)
        abc_service = sdk2.parameters.Integer('ABC service', required=True)
        yp_dcs = sdk2.parameters.List('YP datacenters', required=True)
        awacs_namespace = sdk2.parameters.String('AWACS namespace', required=True)
        yav_awacs_token = sdk2.parameters.YavSecret('YAV secret with AWACS token', required=True)
        yav_arc_token = sdk2.parameters.YavSecret('YAV secret with Arc token', required=True)
        max_inactive_days = sdk2.parameters.Integer('Max days PR was inactive', required=True, default=30)
        config = sdk2.parameters.JSON('Jobs config', required=True, default={
            "jobs": [
                {
                    "awacs_domain": "unstable.example.yandex.net",
                    "stage_templates": [
                        r"^pr-(\d+)-service-1_unstable$",
                        r"^pr-(\d+)-service-2_unstable$",
                        r"^service-3_pullrequest-(\d+)$"
                    ]
                }
            ]
        })
        delay = sdk2.parameters.Integer('Delay between AWACS and YD', required=True, default=900)
        dry_run = sdk2.parameters.Bool('Dry run', default=True, required=True)

    def on_execute(self):
        def nested_dict():
            return collections.defaultdict(nested_dict)

        def http_response(response, decode_json=False):
            """ Get response content """
            if response.status_code != requests.codes.ok:
                sys.exit('Error {} [{}], {}: {}'.format(
                    response.status_code,
                    response.url,
                    response.reason,
                    response.text)
                )
            content = response.json() if decode_json else response.text
            return content

        def request_post(url, post_content, headers=None, decode_json=False):
            """ Make POST request """
            if headers is None:
                headers = {}
            response = requests.post(url, headers=headers, json=post_content, verify=False, timeout=10)
            content = http_response(response, decode_json)
            return content

        def request_get(url, headers=None, query_params=None, decode_json=False):
            """ Make GET request """
            if headers is None:
                headers = {}
            if query_params is None:
                query_params = {}
            response = requests.get(url, headers=headers, params=query_params, verify=False, timeout=10)
            content = http_response(response, decode_json)
            return content

        def download_file(url, filename, headers=None, query_params=None):
            """ Download file """
            if headers is None:
                headers = {}
            if query_params is None:
                query_params = {}
            response = requests.get(url, headers=headers, params=query_params, verify=False, timeout=10)
            if response.status_code != requests.codes.ok:
                sys.exit('Error {} [{}], {}: {}'.format(
                    response.status_code,
                    response.url,
                    response.reason,
                    response.text)
                )
            open(filename, 'wb').write(response.content)
            return response.status_code

        def get_ya_bin(arch, path, search_limit):
            """ Download ya-bin """
            glycine_resource = request_get(
                'https://sandbox.yandex-team.ru/api/v2/resource',
                query_params={
                    'type': 'GLYCINE_RESOURCE',
                    'owner': 'YATOOL',
                    'status': 'RELEASED',
                    'state': 'READY',
                    'limit': search_limit,
                    'offset': 0
                },
                decode_json=True
            )
            resources_url = None
            for item in glycine_resource['items']:
                task_url = item['task']['url']
                task_info = request_get(task_url, decode_json=True)
                if task_info['type'] == 'GLYCINE_BUILD_YA_TOOLS' and task_info['input_parameters']['build_ya'] is True:
                    resources_url = task_info['resources']['url']
                    break
            platform_mapping_url = None
            if resources_url:
                resources_info = request_get(resources_url, decode_json=True)
                for item in resources_info['items']:
                    if item['type'] == 'PLATFORM_MAPPING' and item['attributes']['binary'] == 'devtools/ya/bin/ya-bin':
                        platform_mapping_url = item['http']['proxy']
                        break
            else:
                sys.exit('Could not find "GLYCINE_BUILD_YA_TOOLS" resource')
            if platform_mapping_url:
                platform_mapping = request_get(platform_mapping_url, decode_json=True)
                ya_bin_url = platform_mapping['data'][arch]['url']
                ya_bin_tgz_path = '/tmp/ya-bin.tgz'
                download_file(ya_bin_url, ya_bin_tgz_path)
                tarfile.open(ya_bin_tgz_path).extractall(path)
            else:
                sys.exit('Could not find "PLATFORM_MAPPING" resource')

        def get_yd_stages(yp_token, abc_service, dcs):
            """ Get YD stages """
            headers = {
                'Authorization': 'OAuth ' + yp_token,
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }
            post_content = {
                'object_type': 'pod_set',
                'selector': {
                    'paths': ['/meta/id']
                },
                'filter': {
                    'query': '[/spec/account_id] IN ("abc:service:{}") AND ([/labels/deploy_engine] = "MCRSC" OR [/labels/deploy_engine] = "RSC")'.format(
                        abc_service)
                }
            }
            yd_stages = nested_dict()
            for dc in dcs:
                url = 'https://{}.yp.yandex.net:8443/ObjectService/SelectObjects'.format(dc)
                content = request_post(url, post_content, headers, True)
                if 'results' in content:
                    for result in content['results']:
                        stage, du = result['values'][0].split('.')
                        if du not in yd_stages[stage]:
                            yd_stages[stage][du] = []
                        yd_stages[stage][du].append(dc)
            return yd_stages

        def get_arcadia_pr_status(pr_id, arc_token):
            """ Get Arcadia PR status """
            headers = {
                'Authorization': 'OAuth ' + arc_token,
                'Accept': 'application/json'
            }
            url = 'https://a.yandex-team.ru/api/v1/pull-requests/{}'.format(pr_id)
            query_params = {'fields': 'updated_at,status'}
            content = request_get(url, headers, query_params, True)
            pr_status = content['data']
            return pr_status

        # Fill variables from SB
        yav_yp_token = self.Parameters.yav_yp_token
        abc_service = self.Parameters.abc_service
        yp_dcs = self.Parameters.yp_dcs
        awacs_namespace = self.Parameters.awacs_namespace
        yav_awacs_token = self.Parameters.yav_awacs_token
        yav_arc_token = self.Parameters.yav_arc_token
        max_inactive_days = self.Parameters.max_inactive_days
        config = self.Parameters.config
        delay = self.Parameters.delay
        dry_run = self.Parameters.dry_run

        logging.info('YAV secret with YP token: {}'.format(yav_yp_token))
        logging.info('ABC service: {}'.format(abc_service))
        logging.info('YP datacenters: {}'.format(yp_dcs))
        logging.info('AWACS namespace: {}'.format(awacs_namespace))
        logging.info('YAV secret with AWACS token: {}'.format(yav_awacs_token))
        logging.info('YAV secret with Arc token: {}'.format(yav_arc_token))
        logging.info('Max days PR was inactive: {}'.format(max_inactive_days))
        logging.info('Jobs config: {}'.format(config))
        logging.info('Delay between AWACS and YD: {}'.format(delay))
        logging.info('Dry run: {}'.format(dry_run))

        # Get AWACS token from YAV
        logging.info('Getting YAV secrets')
        yp_token = yav_yp_token.value()
        awacs_token = yav_awacs_token.value()
        arc_token = yav_arc_token.value()

        # Get YD stages
        yd_stages = get_yd_stages(yp_token, abc_service, yp_dcs)
        logging.info('Current stages ({}):\n{}'.format(len(yd_stages), json.dumps(yd_stages, indent=2)))

        unstables_to_delete = dict()
        prs_status = dict()
        current_date = pytz.utc.localize(datetime.datetime.utcnow())

        # Execute jobs
        for job in config['jobs']:
            awacs_domain = job['awacs_domain']
            stage_templates = job['stage_templates']

            logging.info('AWACS domain: {}'.format(awacs_domain))
            logging.info('Stage templates: {}'.format(stage_templates))

            # Find stages with closed PR
            for stage in yd_stages.keys():
                # Check stage templates
                for template in stage_templates:
                    match = re.search(template, stage)
                    if match:
                        pr_id = match.group(1)
                        # Check PR status
                        if pr_id not in prs_status:
                            prs_status[pr_id] = get_arcadia_pr_status(pr_id, arc_token)
                            updated_at = parser.parse(prs_status[pr_id]['updated_at'])
                            prs_status[pr_id]['updated_days_ago'] = (current_date - updated_at).days
                        # Add stage to delete
                        if (prs_status[pr_id]['status'] in ['merged', 'discarded']) or (
                                prs_status[pr_id]['updated_days_ago'] >= max_inactive_days):
                            logging.info(
                                'Stage "{}" matched template "{}", PR ID: {}, status: {}, updated days ago: {}'.format(
                                    stage, template, pr_id, prs_status[pr_id]['status'],
                                    prs_status[pr_id]['updated_days_ago']))
                            if awacs_domain not in unstables_to_delete:
                                unstables_to_delete[awacs_domain] = []
                            unstables_to_delete[awacs_domain].append(stage)
                        break

        logging.info('PRs status ({}):\n{}'.format(len(prs_status), json.dumps(prs_status, indent=2)))
        logging.info('Unstables to delete:\n{}'.format(json.dumps(unstables_to_delete, indent=2)))

        awacs_stage_inserter = 'awacs_stage_inserter'
        ya_bin = 'ya-bin'
        if (not dry_run) and unstables_to_delete:
            # Download 'awacs_stage_inserter'
            logging.info("Downloading 'awacs_stage_inserter'")
            awacs_stage_inserter_resource = sdk2.Resource['AWACS_STAGE_INSERTER'].find(
                type='YA_MAKE',
                state='READY',
                status='RELEASED',
                owner='KINOPOISK',
                arch='linux'
            ).first()
            if not awacs_stage_inserter_resource:
                sys.exit("Could not find 'awacs_stage_inserter' resource")
            awacs_stage_inserter = str(sdk2.ResourceData(
                awacs_stage_inserter_resource).path) + '/awacs_stage_inserter/awacs_stage_inserter'

            # Download 'ya-bin'
            logging.info("Downloading 'ya-bin'")
            ya_bin = '/tmp/ya-bin'
            get_ya_bin(arch='linux_musl', path='/tmp', search_limit=10)

            # Set environment variables
            logging.info('Setting environment variables')
            os.environ['AWACS_TOKEN'] = awacs_token
            os.environ['DCTL_YP_TOKEN'] = yp_token

        # Remove unstables from AWACS
        for awacs_domain in unstables_to_delete.keys():
            for stage in unstables_to_delete[awacs_domain]:
                logging.info("Removing unstable from AWACS: '{}'".format(stage))
                # Remove all deploy units from AWACS
                for du in yd_stages[stage].keys():
                    cmd = [awacs_stage_inserter,
                           '--action', 'remove_unstable',
                           '--namespace', awacs_namespace,
                           '--endpoint-set', '{}.{}'.format(stage, du),
                           '--domain', awacs_domain,
                           '--datacenters'] + yd_stages[stage][du]
                    if dry_run:
                        logging.info('DRY RUN: {}'.format(cmd))
                    else:
                        run_process(cmd)

        # Wait for AWACS balancers to deploy
        if (not dry_run) and unstables_to_delete:
            logging.info('Sleeping for {} seconds'.format(delay))
            time.sleep(delay)

        # Remove unstables from YD
        for awacs_domain in unstables_to_delete.keys():
            for stage in unstables_to_delete[awacs_domain]:
                logging.info("Removing unstable from YD: '{}'".format(stage))
                cmd = [ya_bin, 'dctl', 'remove', 'stage', stage]
                if dry_run:
                    logging.info('DRY RUN: {}'.format(cmd))
                else:
                    run_process(cmd)
