import collections
import json
import logging
import os
import sys
import tarfile
import time

import requests
import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.sandboxsdk.process import run_process


class YdAwacsGithubRemoveUnstables(sdk2.Task):
    """ Remove GitHub unstables from YD and AWACS """

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 3072
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):

        yav_yp_token = sdk2.parameters.YavSecret('YAV secret with YP token', required=True)
        abc_service = sdk2.parameters.Integer('ABC service', required=True)
        yp_dcs = sdk2.parameters.List('YP datacenters', required=True)
        awacs_namespace = sdk2.parameters.String('AWACS namespace', required=True)
        yav_awacs_token = sdk2.parameters.YavSecret('YAV secret with AWACS token', required=True)
        yav_github_enterprise_token = sdk2.parameters.YavSecret('YAV secret with enterprise GitHub token',
                                                                required=False)
        yav_github_public_token = sdk2.parameters.YavSecret('YAV secret with public GitHub token', required=False)
        yav_bitbucket_enterprise_token = sdk2.parameters.YavSecret('YAV secret with enterprise BitBucket token',
                                                                   required=False)
        config = sdk2.parameters.JSON('Jobs config', required=True, default={
            "jobs": [
                {
                    "awacs_domain": "unstable.example.yandex.net",
                    "github_type": "enterprise|public|bb-enterprise",
                    "github_repo": "user/repo",
                    "stage_template": "pr-{}-stage_unstable"
                }
            ]
        })
        delay = sdk2.parameters.Integer('Delay between AWACS and YD', required=True, default=900)
        dry_run = sdk2.parameters.Bool('Dry run', default=True, required=True)

    def on_execute(self):
        def nested_dict():
            return collections.defaultdict(nested_dict)

        def http_response(response, decode_json=False):
            """ Get response content """
            if response.status_code != requests.codes.ok:
                sys.exit('Error {} [{}], {}: {}'.format(
                    response.status_code,
                    response.url,
                    response.reason,
                    response.text)
                )
            content = response.json() if decode_json else response.text
            return content

        def request_post(url, post_content, headers=None, decode_json=False):
            """ Make POST request """
            if headers is None:
                headers = {}
            response = requests.post(url, headers=headers, json=post_content, verify=False, timeout=10)
            content = http_response(response, decode_json)
            return content

        def request_get(url, headers=None, query_params=None, decode_json=False):
            """ Make GET request """
            if headers is None:
                headers = {}
            if query_params is None:
                query_params = {}
            response = requests.get(url, headers=headers, params=query_params, verify=False, timeout=10)
            content = http_response(response, decode_json)
            return content

        def download_file(url, filename, headers=None, query_params=None):
            """ Download file """
            if headers is None:
                headers = {}
            if query_params is None:
                query_params = {}
            response = requests.get(url, headers=headers, params=query_params, verify=False, timeout=10)
            if response.status_code != requests.codes.ok:
                sys.exit('Error {} [{}], {}: {}'.format(
                    response.status_code,
                    response.url,
                    response.reason,
                    response.text)
                )
            open(filename, 'wb').write(response.content)
            return response.status_code

        def get_ya_bin(arch, path, search_limit):
            """ Download ya-bin """
            glycine_resource = request_get(
                'https://sandbox.yandex-team.ru/api/v2/resource',
                query_params={
                    'type': 'GLYCINE_RESOURCE',
                    'owner': 'YATOOL',
                    'status': 'RELEASED',
                    'state': 'READY',
                    'limit': search_limit,
                    'offset': 0
                },
                decode_json=True
            )
            resources_url = None
            for item in glycine_resource['items']:
                task_url = item['task']['url']
                task_info = request_get(task_url, decode_json=True)
                if task_info['type'] == 'GLYCINE_BUILD_YA_TOOLS' and task_info['input_parameters']['build_ya'] is True:
                    resources_url = task_info['resources']['url']
                    break
            platform_mapping_url = None
            if resources_url:
                resources_info = request_get(resources_url, decode_json=True)
                for item in resources_info['items']:
                    if item['type'] == 'PLATFORM_MAPPING' and item['attributes']['binary'] == 'devtools/ya/bin/ya-bin':
                        platform_mapping_url = item['http']['proxy']
                        break
            else:
                sys.exit('Could not find "GLYCINE_BUILD_YA_TOOLS" resource')
            if platform_mapping_url:
                platform_mapping = request_get(platform_mapping_url, decode_json=True)
                ya_bin_url = platform_mapping['data'][arch]['url']
                ya_bin_tgz_path = '/tmp/ya-bin.tgz'
                download_file(ya_bin_url, ya_bin_tgz_path)
                tarfile.open(ya_bin_tgz_path).extractall(path)
            else:
                sys.exit('Could not find "PLATFORM_MAPPING" resource')

        def arrays_intersection(array_1, array_2):
            """ Arrays intersection """
            return [value for value in array_2 if value in array_1]

        def get_yd_stages(yp_token, abc_service, dcs):
            """ Get YD stages """
            headers = {
                'Authorization': 'OAuth ' + yp_token,
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }
            post_content = {
                'object_type': 'pod_set',
                'selector': {
                    'paths': ['/meta/id']
                },
                'filter': {
                    'query': '[/spec/account_id] IN ("abc:service:{}") AND ([/labels/deploy_engine] = "MCRSC" OR [/labels/deploy_engine] = "RSC")'.format(
                        abc_service)
                }
            }
            yd_stages = nested_dict()
            for dc in dcs:
                url = 'https://{}.yp.yandex.net:8443/ObjectService/SelectObjects'.format(dc)
                content = request_post(url, post_content, headers, True)
                if 'results' in content:
                    for result in content['results']:
                        stage, du = result['values'][0].split('.')
                        if du not in yd_stages[stage]:
                            yd_stages[stage][du] = []
                        yd_stages[stage][du].append(dc)
            return yd_stages

        def get_github_issues(github_repo, github_type='enterprise', github_token=None):
            """ Get GitHub issues """
            github_url = {
                'public': 'https://api.github.com',
                'enterprise': 'https://github.yandex-team.ru/api/v3',
                'bb-enterprise': 'https://bb.yandex-team.ru/rest/api/1.0'
            }
            github_issues = list()
            if github_type in ['enterprise', 'public']:
                url = '{}/repos/{}/issues'.format(github_url[github_type], github_repo)
                search_query = {
                    'state': 'closed',
                    'sort': 'updated',
                    'per_page': 100,
                    'direction': 'desc'
                }
                headers = {} if github_token is None else {'Authorization': 'token ' + github_token}
                content = request_get(url, headers, search_query, True)
                github_issues = [x['number'] for x in content]
            elif github_type == 'bb-enterprise':
                project_key, repository_slug = github_repo.split('/')
                url = '{}/projects/{}/repos/{}/pull-requests'.format(github_url[github_type],
                                                                     project_key,
                                                                     repository_slug)
                search_query = {
                    'state': 'ALL',
                    'order': 'NEWEST',
                    'limit': 100
                }
                headers = {} if github_token is None else {'Authorization': 'Bearer ' + github_token}
                content = request_get(url, headers, search_query, True)
                github_issues = [x['id'] for x in content['values'] if x['state'] in ['MERGED', 'DECLINED']]
            return github_issues

        # Fill variables from SB
        yav_yp_token = self.Parameters.yav_yp_token
        abc_service = self.Parameters.abc_service
        yp_dcs = self.Parameters.yp_dcs
        awacs_namespace = self.Parameters.awacs_namespace
        yav_awacs_token = self.Parameters.yav_awacs_token
        yav_github_enterprise_token = self.Parameters.yav_github_enterprise_token
        yav_github_public_token = self.Parameters.yav_github_public_token
        yav_bitbucket_enterprise_token = self.Parameters.yav_bitbucket_enterprise_token
        config = self.Parameters.config
        dry_run = self.Parameters.dry_run
        delay = self.Parameters.delay

        logging.info('YAV secret with YP token: {}'.format(yav_yp_token))
        logging.info('ABC service: {}'.format(abc_service))
        logging.info('YP datacenters: {}'.format(yp_dcs))
        logging.info('AWACS namespace: {}'.format(awacs_namespace))
        logging.info('YAV secret with AWACS token: {}'.format(yav_awacs_token))
        logging.info('YAV secret with enterprise GitHub token: {}'.format(yav_github_enterprise_token))
        logging.info('YAV secret with public GitHub token: {}'.format(yav_github_public_token))
        logging.info('YAV secret with enterprise BitBucket token: {}'.format(yav_bitbucket_enterprise_token))
        logging.info('Jobs config: {}'.format(config))
        logging.info('Delay between AWACS and YD: {}'.format(delay))
        logging.info('Dry run: {}'.format(dry_run))

        # Get AWACS token from YAV
        logging.info('Getting YAV secrets')
        yp_token = yav_yp_token.value()
        awacs_token = yav_awacs_token.value()
        github_enterprise_token = yav_github_enterprise_token.value() if yav_github_enterprise_token is not None else None
        github_public_token = yav_github_public_token.value() if yav_github_public_token is not None else None
        bitbucket_enterprise_token = yav_bitbucket_enterprise_token.value() if yav_bitbucket_enterprise_token is not None else None

        # Get YD stages
        yd_stages = get_yd_stages(yp_token, abc_service, yp_dcs)
        logging.info('Current stages ({}):\n{}'.format(len(yd_stages), json.dumps(yd_stages, indent=2)))

        unstables_to_delete = dict()

        # Execute jobs
        for job in config['jobs']:
            awacs_domain = job['awacs_domain']
            github_type = job['github_type'] if 'github_type' in job else 'enterprise'
            github_repo = job['github_repo']
            stage_template = job['stage_template']

            logging.info('AWACS domain: {}'.format(awacs_domain))
            logging.info('GitHub type: {}'.format(github_type))
            logging.info('GitHub repo: {}'.format(github_repo))
            logging.info('Stage template: {}'.format(stage_template))

            if github_type == 'bb-enterprise':
                github_token = bitbucket_enterprise_token
            elif github_type == 'public':
                github_token = github_public_token
            else:
                github_token = github_enterprise_token

            # Get closed issues from GitHub
            github_closed_issues = [stage_template.format(x) for x in get_github_issues(
                github_repo,
                github_type,
                github_token
            )]
            logging.info(
                'Closed issues ({}):\n  {}'.format(len(github_closed_issues), '\n  '.join(github_closed_issues)))

            # Calculate unnecessary stages
            stages_with_closed_issues = arrays_intersection(yd_stages.keys(), github_closed_issues)
            logging.info('Unnecessary stages ({}):\n  {}'.format(len(stages_with_closed_issues),
                                                                 '\n  '.join(stages_with_closed_issues)))

            # Add stages to delete
            for stage in stages_with_closed_issues:
                if awacs_domain not in unstables_to_delete:
                    unstables_to_delete[awacs_domain] = []
                unstables_to_delete[awacs_domain].append(stage)

        logging.info('Unstables to delete:\n{}'.format(json.dumps(unstables_to_delete, indent=2)))

        awacs_stage_inserter = 'awacs_stage_inserter'
        ya_bin = 'ya-bin'
        if (not dry_run) and unstables_to_delete:
            # Download 'awacs_stage_inserter'
            logging.info("Downloading 'awacs_stage_inserter'")
            awacs_stage_inserter_resource = sdk2.Resource['AWACS_STAGE_INSERTER'].find(
                type='YA_MAKE',
                state='READY',
                status='RELEASED',
                owner='KINOPOISK',
                arch='linux'
            ).first()
            if not awacs_stage_inserter_resource:
                sys.exit("Could not find 'awacs_stage_inserter' resource")
            awacs_stage_inserter = str(sdk2.ResourceData(
                awacs_stage_inserter_resource).path) + '/awacs_stage_inserter/awacs_stage_inserter'

            # Download 'ya-bin'
            logging.info("Downloading 'ya-bin'")
            ya_bin = '/tmp/ya-bin'
            get_ya_bin(arch='linux_musl', path='/tmp', search_limit=10)

            # Set environment variables
            logging.info('Setting environment variables')
            os.environ['AWACS_TOKEN'] = awacs_token
            os.environ['DCTL_YP_TOKEN'] = yp_token

        # Remove unstables from AWACS
        for awacs_domain in unstables_to_delete.keys():
            for stage in unstables_to_delete[awacs_domain]:
                logging.info("Removing unstable from AWACS: '{}'".format(stage))
                # Remove all deploy units from AWACS
                for du in yd_stages[stage].keys():
                    cmd = [awacs_stage_inserter,
                           '--action', 'remove_unstable',
                           '--namespace', awacs_namespace,
                           '--endpoint-set', '{}.{}'.format(stage, du),
                           '--domain', awacs_domain,
                           '--datacenters'] + yd_stages[stage][du]
                    if dry_run:
                        logging.info('DRY RUN: {}'.format(cmd))
                    else:
                        run_process(cmd)

        # Wait for AWACS balancers to deploy
        if (not dry_run) and unstables_to_delete:
            logging.info('Sleeping for {} seconds'.format(delay))
            time.sleep(delay)

        # Remove unstables from YD
        for awacs_domain in unstables_to_delete.keys():
            for stage in unstables_to_delete[awacs_domain]:
                logging.info("Removing unstable from YD: '{}'".format(stage))
                cmd = [ya_bin, 'dctl', 'remove', 'stage', stage]
                if dry_run:
                    logging.info('DRY RUN: {}'.format(cmd))
                else:
                    run_process(cmd)
