import os
import logging
from sandbox import sdk2, common
from sandbox.projects.media.resources import JupyEnvironment
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.media.admins.common.utils import git_clone as clone, git_co
from sandbox.projects.media.admins.common.utils import obfuscate, nyan_notify
from sandbox.projects.media.admins.common.utils import run_cmd, get_yav_secret, get_sb_secret


class AutoJupy(sdk2.Task):
    """
    Automatically apply merged monitoring configs with jupycli
    About jupycli: https://wiki.yandex-team.ru/muz/admin/jupy/
    """
    log = logging.getLogger("AutoJupy")
    dryrun = True
    state = {}
    yav_secret = None
    pr_payload = None

    class Parameters(sdk2.Parameters):
        """
        Form parameters
        """
        container = sdk2.parameters.Container('Container', required=True,
                                              platform='linux_ubuntu_16.04_xenial')

        with sdk2.parameters.Group('Repository') as params:
            repo_name = sdk2.parameters.String("Repository link", required=False)
            target = sdk2.parameters.String("target test", required=False, default="cloned_repo")
            notify_chat = sdk2.parameters.String("Notify to chat name", required=False)
            yav_token = sdk2.parameters.String("OAuth token Vault name for YAV",
                                               required=False,
                                               default="media.secure.robot-media-salt.yav.token")

    class Requirements(sdk2.Task.Requirements):
        """
        binary and others requirements
        """
        privileged = True
        environments = (
            JupyEnvironment(),
            PipEnvironment('GitPython', version='2.1.11', use_user=False),
        )

    def init_jupy_env(self):
        """
        Init OAuth token for jupy authenticatin in juggler
        :return: None
        """
        yav_token = get_sb_secret(self.Parameters.yav_token, self.owner)
        if not yav_token:
            raise Exception("Failed to get secret data from Sandbox Vault.")
        juggler_token = get_yav_secret(self.yav_secret, yav_token, key='juggler_oauth_token')
        if not juggler_token:
            raise Exception("Failed to get secret from YAV.")
        juggler_token = juggler_token.strip()
        obfuscated_jtoken = obfuscate(juggler_token, 3, 1)
        self.log.debug("Got token %s", obfuscated_jtoken)
        self.log.info("Set jupycli environment.")
        self.log.info("OS ENV PATH: %s", os.environ["PATH"])
        os.environ.update({'JUPY_JUGGLER_OAUTH_TOKEN': juggler_token})
        self.log.info("Added env var JUPY_JUGGLER_OAUTH_TOKEN: %s", obfuscated_jtoken)
        self.log.info("Done.")

    def jupycli(self, dryrun=True):
        """
        Run jupycli
        :param dryrun: if True run with -n (dry-run)
        :return: bool
        """
        self.init_jupy_env()
        self.log.info("Jupy path: %s", self.Requirements.environments[0].path)
        jupycli = "jupycli"
        cmd = jupycli + " -f -d -n" if dryrun else jupycli + " -f -d"
        res = run_cmd(cmd)
        if res[0] != 0:
            self.log.error("Failed to apply configs: %s", res[1])
            return False

        self.log.info("Jupy output: %s", res[1])
        return True

    def get_changes(self):
        """
        Get changes fro pr diff_url attribute. When PR in closed state there are no attr
        changed_files and needed workaround.
        :return: changed files list
        """
        import requests
        import re
        try:
            diff_url = self.pr_payload['pull_request']['diff_url']
            diff_url = diff_url.replace("github.yandex-team.ru/", "github.yandex-team.ru/raw/")
        except KeyError:
            self.log.error("Failed to get changed url.")
            return None
        self.log.info("Got diff url: %s", diff_url)
        try:
            data = requests.get(diff_url)
        except (requests.ConnectionError, requests.HTTPError) as err:
            self.log.error("Failed to get changes from %s: %s", diff_url, err)
            return None

        changed_files = re.findall(r"diff --git a/([.\w/]+)", data.content)

        return changed_files

    def apply_projects(self, dryrun=True):
        """
        Apply projects monitoring
        :param dryrun: run jupy with dryrun or not
        :return: dict (project: state)
        """
        root_dir = os.path.abspath(os.path.curdir) + "/" + self.Parameters.target
        projects = self.get_projects()
        root_dir = root_dir + '/juggler'
        self.log.info("Set root dir: %s", root_dir)

        if not projects:
            self.log.info("No changed projects found in PR changes. Try to apply %s", root_dir)
            return None

        state = {}
        for project in projects:
            if project is None:
                self.log.debug("Changed files are in root juggler project.")
                project = ''
            work_dir = "{}/{}".format(root_dir, project)
            self.log.debug("Set work dir to %s", work_dir)
            os.chdir(work_dir)
            self.log.info("Apply project %s", project)
            state[os.path.basename(work_dir)] = self.jupycli(dryrun=dryrun)
            os.chdir(root_dir)
            self.log.info("Done.")

        return state

    def get_projects(self):
        """
        Get project names for apply configs. Based on repository structure like this:
        repo_root/jugggler/project_name/configs/* or repo_root/jugggler/configs
        :return: projects list or None if no projects found
        """
        import re
        changed_files = self.get_changes()

        if not changed_files:
            self.log.info("Failed to get changes from github diff url")
            return None

        self.log.info("Changed files: %s", changed_files)

        projects = []
        for fname in changed_files:
            try:
                project = re.match(r"(juggler)/(\w+)?/?(configs[\w/.]+)", fname).group(2)
                if project not in projects:
                    projects.append(project)
            except AttributeError:
                self.log.warning("Failed to get project for file %s. Check repo structure.", fname)
                pass
        if projects:
            self.log.debug("Found changed monitoring projects: %s", projects)
            return projects
        else:
            return None

    def on_execute(self):

        target = self.Parameters.target
        self.dryrun = False

        if "_debug" in self.Context.__values__:
            self.log.info("Debug context: %s(dry run mode)", self.Context.__values__['_debug'])
            self.dryrun = bool(self.Context.__values__['_debug'])
            self.log.info("Got dry run value %s", self.dryrun)

        if "secret" in self.Context.__values__:
            self.yav_secret = self.Context.__values__['secret']
            self.log.info("Got YAV secret %s", self.yav_secret)
        else:
            raise Exception("Can not process without YAV secret with OAuth token")

        try:
            self.pr_payload = self.Context.__values__['githubEvent']['payload']
        except ValueError:
            self.log.error("Failed to get PR payload.")
            raise

        vcs_repo = self.pr_payload['repository']['clone_url']
        self.log.info("Got repository: %s", vcs_repo)
        base = self.pr_payload['pull_request']['base']['ref']
        head = self.pr_payload['pull_request']['head']['ref']
        commit_autor = self.pr_payload['sender']['login']
        pr_url = self.pr_payload['pull_request']['html_url']

        self.log.info("PR: %s -> %s", head, base)
        try:
            action = self.pr_payload['action']
            self.log.info("Got action %s", action)
        except KeyError:
            self.log.error("Failed to get action from PR")
            raise

        if action in ['synchronize', 'opened', 'reopened']:
            git_co(head, vcs_repo, target)
            self.dryrun = True
        elif action in ['closed']:
            clone(vcs_repo, target)
            self.log.info("PR Action: %s", action)
        else:
            self.log.info("Other PR Action: %s", action)

        self.log.info("Dry run mode: %s", self.dryrun)
        state = self.apply_projects(dryrun=self.dryrun)
        if not state:
            self.log.info("No changes in juggler configs. Nothing to do.")
            return None
        if self.Parameters.notify_chat:
            info = "Please look at jupy apply result.\n" \
                   "Commit Author: {}\n" \
                   "PR: {}\n" \
                   "Sandbox task: {}/task/{}/view\n" \
                   "Apply state: {}\n".format(commit_autor, pr_url, common.utils.server_url(),
                                              self.id, state)
            nyan_notify(info, self.Parameters.notify_chat)
