"""
Common utilities
"""
import os
import subprocess
from sandbox.sandboxsdk.process import run_process, SandboxSubprocessError
import logging
import base64

from sandbox import sdk2

LOG = logging.getLogger("utils")


def parse_secret(secret):
    """
    parse secret string defined as secret[key]
    :param secret: yav secret in format secret[key]
    :return: tuple secret, key
    """
    try:
        sec, key = secret.split("[")
    except ValueError:
        LOG.error("Can't parse secret %s. Valid format: some_secret[some_key]", secret)
        raise

    key = key.strip("]").strip()
    sec = sec.strip()
    return sec, key


# noinspection PyBroadException
def is_base64encoded(s):
    """
    detect base64 encoded str
    :param s: str
    :return: bool True if base64 encoded
    """
    try:
        # if s is not encoded usual string, we expect the exception
        # 1st try this(it doesn't work with cyrillic!!)
        base64.b64decode(s).encode("ascii")
        # if 1st passes try return this
        return base64.b64encode(base64.b64decode(s)) == s
    except:
        return False


def git_clone(repository, destination):
    """
    Clone git repository into local destination
    :param repository: git repository url
    :param destination: local path where to clone
    :return: None
    """
    os.path.exists(destination) or os.makedirs(destination)
    cmd = ['git', 'clone', repository, destination]
    try:
        run_process(cmd)
    except SandboxSubprocessError as err:
        LOG.error(err)
        raise


def git_co(bname, repo, dst):
    """
    Git checkout to branch bname
    :param bname: branch name
    :param repo: reposutory url
    :param dst: where to clone
    :return: None
    """
    os.path.exists(dst) or os.makedirs(dst)
    LOG.info("Try to clone repo %s", repo)
    git_clone(repo, dst)
    LOG.info("Got it!")
    work_dir = os.path.abspath(os.path.curdir)
    LOG.info("Change dir to %s", dst)
    os.chdir(dst)
    LOG.info("Switch to branch %s...", bname)
    cmd = ['git', 'checkout', bname]
    try:
        run_process(cmd)
    except SandboxSubprocessError as err:
        LOG.error(err)
        raise
    LOG.info("Branch switched.")
    LOG.info("Pull PR...")
    cmd = ['git', 'pull', 'origin', bname]
    try:
        run_process(cmd)
    except SandboxSubprocessError as err:
        LOG.error(err)
        raise
    LOG.info("Change dir to working dir root %s", work_dir)
    os.chdir(work_dir)
    LOG.info("Done.")
    return dst


def clean(path):
    """
    clean all data in path
    :param path: path on FS
    """
    if path is not None:
        if os.path.exists(path):
            protect = ['/', '/usr', '/var', '/lib']
            if path in protect:
                return None
            for root, dirs, files in os.walk(path, topdown=False):
                for name in files:
                    os.remove(os.path.join(root, name))
                for name in dirs:
                    os.rmdir(os.path.join(root, name))
            LOG.debug("Clean complete for path %s", path)
            return None
        else:
            LOG.debug("%s does not exists, clean is not needed.", path)
            return None
    else:
        LOG.debug("Got empty path for clean.")
        return None


def run_cmd(cmd):
    """
    Run command
    :param cmd:
    :return:
    """
    try:
        result = subprocess.check_output(cmd.split(), stderr=subprocess.STDOUT)
        return 0, result
    except subprocess.CalledProcessError as err:
        LOG.error(err)
        LOG.error(err.output)
        return err.returncode, err


def nyan_notify(msg, chat):
    """
    Send message to chat over nyan bot
    :param msg: message
    :param chat: chat name as project/chat from u/kotoaminka. Example: kp/kinomonga
    :return: bool
    """
    import requests
    import json
    data = {
        "text": msg
    }
    certs = '/etc/ssl/certs/ca-certificates.crt'
    headers = {"Content-Type": "application/json", "Accept": "application/json"}
    url = "https://api.nbt.media.yandex.net/v1/proxy/telegram/" + chat
    LOG.info("Generated request data for neko: url %s, headers %s, text %s", url, headers, msg)
    try:
        resp = requests.post(url, json.dumps(data), headers=headers, verify=certs)
        LOG.info("Got response %s", resp)
    except (requests.ConnectionError, requests.RequestException, requests.HTTPError) as err:
        LOG.error("Failed to send message to chat %s, %s", chat, err)
        return False

    return True


def obfuscate(secret, begin=0, end=0):
    """
    Obfuscate secret str
    :param secret: secret str
    :param begin: number of symbols to show at the beginning ;
    :param end: number of symbols to show at the end
    :return: obfuscated str. If begin == end == 0 return '******'
    """
    if begin == 0 and end == 0:
        return "*************"
    elif begin == 0:
        return "*************" + secret[-end:]
    elif end == 0:
        return secret[:begin] + "*************"

    return secret[:begin] + "*************" + secret[-end:]


def load_hook_data(payload):
    """
    Load json webhook data
    :return: dict
    """
    import json

    try:
        data = json.loads(payload)
    except KeyError:
        LOG.error("Failed to load payload as json")
        data = None

    return data


def get_sb_secret(secret_alias, owner):
    """
            Retrieves secret from Sandbox vault
            :param secret_alias: alias of the secret to retrieve
            :param owner: owner of the secret
            :return: secret data
            """
    LOG.info("Get Sandbox Vault secret alias %s", secret_alias)
    return sdk2.Vault.data(owner, secret_alias)


def get_yav_secret(secret_id, oauth_token, key=None):
    """
            Retrieves secret from YAV via HTTP
            :param oauth_token: oauth token to YAV for HTTP requests
            :param secret_id: id of the secret to retrieve
            :param key: key to get fron yav secret data
            :param decode: decode key data(default: True)
            :return:
    """
    LOG.info("Get YAV secret %s", secret_id)

    import requests
    import base64

    def is_base64encoded(s):
        try:
            # if s is not encoded usual string, we expect the exception
            # 1st try this(it doesn't work with cyrillic!!)
            base64.b64decode(s).encode("ascii")
            # if 1st passes try return this
            return base64.b64encode(base64.b64decode(s)) == s
        except:
            return False

    api_url = 'https://vault-api.passport.yandex.net/1/versions/{}'.format(secret_id)
    secret = requests.get(api_url, headers={'Authorization': 'OAuth {}'.format(oauth_token)},
                          verify=False).json()
    if secret['status'] != 'ok':
        raise Exception('Error on getting YAV secret data: {0}'.format(secret['message']))

    LOG.info("Got secrets from %s", secret_id)
    secret_data = {}
    if key is not None:
        for el in secret['version']['value']:
            if el['key'] == key:
                if is_base64encoded(el['value']):
                    LOG.debug("Secret %s key %s encoded: True, need to decode.", secret_id, key)
                    secret_data = base64.b64decode(el['value'])
                else:
                    LOG.debug("Secret %s key %s encoded: False, return as is.", secret_id, key)
                    secret_data = el['value']

                return secret_data

    for el in secret['version']['value']:
        if is_base64encoded(el['value']):
            secret_data.update({el['key']: base64.b64decode(el['value'])})
        else:
            secret_data.update({el['key']: el['value']})

    return secret_data
