# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox.sdk2.task import ctt
from sandbox.projects.media.admins.mysqlcopydb import MysqlCopyDB
from sandbox.projects.media.admins.mongodump import MongoDumpRestore
import logging


class CopyDbExecuter(sdk2.Task):
    """
    Запускаем переналивку баз подзадачами по очереди, ожидаем завершения.
    Поддерживаются mongodb и mysql(таски: MYSQL_COPY_DB и MONGO_DUMP_RESTORE)
    Дока: https://wiki.yandex-team.ru/users/chrono/CopyDatabases/
    """
    class Requirements(sdk2.Task.Requirements):
        """
        binary and others requirements
        """
        privileged = True

    class Parameters(sdk2.Task.Parameters):
        """
        Form parameters
        """
        with sdk2.parameters.Group('Repo') as params:
            megaconfig = sdk2.parameters.JSON("What Config", description="JSON конфиг переливок")

    log = logging.getLogger("Nalivochnaya")

    def get_params(self, name):
        params = dict()
        conf = self.Parameters.megaconfig
        if 'CopyFrom' in conf[name]:
            source_id = conf[name]['CopyFrom']
            type = conf[name]['type']
            self.log.info("Copy task parameters from task %s", source_id)
            if type == 'MYSQL_COPY_DB':
                found = sdk2.Task.find(task_type=MysqlCopyDB, id=source_id)
            elif type == 'MONGO_DUMP_RESTORE':
                found = sdk2.Task.find(task_type=MongoDumpRestore, id=source_id)
            else:
                return None
            if found:
                found_task = found.first()
                params = dict(found_task.Parameters)
            else:
                for key, value in conf[name]['config'].items():
                    params.update({key: value})
        else:
            for key, value in conf[name]['config'].items():
                self.log.debug("Copy parameter: %s: %s", key, value)
                params.update({key: value})
        return params

    def create_children(self):
        config = dict(self.Parameters.megaconfig)
        children = []
        if config:
            self.log.info(config)
            for task in sorted(config.keys()):
                self.log.info(config[task])
                self.log.info("Create subtask %s", task)
                if config[task]['type'] == 'MYSQL_COPY_DB':
                    params = self.get_params(task)
                    self.log.info("Got parameters {}".format(dict(params)))
                    subtask = MysqlCopyDB(self, **params)
                elif config[task]['type'] == 'MONGO_DUMP_RESTORE':
                    params = self.get_params(task)
                    self.log.info("Got parameters {}".format(dict(params)))
                    subtask = MongoDumpRestore(self, **params)
                else:
                    raise Exception("Unknown task type in config")
                self.log.info("Created task type %s", task)
                self.log.info("Subtask parameters was set with %s", dict(subtask.Parameters))
                subtask.save()
                children.append(subtask)
        return children

    def on_execute(self):

        with self.memoize_stage.create_subtasks_once:
            subtasks = self.create_children()
            self.Context.subtasks = [task.id for task in reversed(subtasks)]
            self.Context.save()

        if self.Context.subtasks:
            subtask_id = self.Context.subtasks.pop()
            self.log.info("Got new subtask to run: %s", subtask_id)
            self.log.info("Context saved. Remain tasks: %s", self.Context.subtasks)
            sdk2.Task[subtask_id].enqueue()
            self.Context.save()
            raise sdk2.WaitTask([subtask_id], ctt.Status.Group.FINISH + ctt.Status.Group.BREAK)
        else:
            self.log.info("All subtasks was finished.")
